<?php

namespace App\Services;

use App\Models\TotalCache;
use App\Models\ShipmentLine;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrders;
use App\Models\Shipment;
use App\Models\CustomerOrderLineQuantities;
use App\Models\ShipmentLineSizes;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class TotalCacheService
{
    /**
     * Get cached totals for an entity.
     */
    public function get(string $entityType, int $entityId, string $cacheKey): ?array
    {
        $cache = TotalCache::forEntity($entityType, $entityId)
            ->where('cache_key', $cacheKey)
            ->fresh()
            ->first();

        return $cache ? $cache->cached_data : null;
    }

    /**
     * Set cached totals for an entity.
     */
    public function set(string $entityType, int $entityId, string $cacheKey, array $data): void
    {
        TotalCache::updateOrCreate(
            [
                'entity_type' => $entityType,
                'entity_id' => $entityId,
                'cache_key' => $cacheKey,
            ],
            [
                'cached_data' => $data,
                'fresh_at' => now(),
            ]
        );
    }

    /**
     * Invalidate cache for a specific entity.
     */
    public function invalidateEntity(string $entityType, int $entityId, ?string $cacheKey = null): void
    {
        $query = TotalCache::forEntity($entityType, $entityId);
        
        if ($cacheKey) {
            $query->where('cache_key', $cacheKey);
        }
        
        $query->update(['fresh_at' => null]);
    }

    /**
     * Invalidate cache for multiple entities.
     */
    public function invalidateEntities(string $entityType, array $entityIds, ?string $cacheKey = null): void
    {
        if (empty($entityIds)) {
            return;
        }

        $query = TotalCache::forEntityType($entityType)
            ->whereIn('entity_id', $entityIds);
        
        if ($cacheKey) {
            $query->where('cache_key', $cacheKey);
        }
        
        $query->update(['fresh_at' => null]);
    }

    /**
     * Invalidate all cache entries for a specific cache key.
     */
    public function invalidateCacheKey(string $cacheKey): void
    {
        TotalCache::where('cache_key', $cacheKey)
            ->update(['fresh_at' => null]);
    }

    /**
     * Warm up totals for a shipment line.
     */
    public function warmupShipmentLine(ShipmentLine $shipmentLine): void
    {
        try {
            // Calculate totals using the existing accessor logic
            $totals = $this->calculateShipmentLineTotals($shipmentLine);
            
            $this->set('shipment_line', $shipmentLine->id, 'prices', $totals);
            
            Log::info("Warmed up totals for shipment line {$shipmentLine->id}");
        } catch (\Exception $e) {
            Log::error("Failed to warm up totals for shipment line {$shipmentLine->id}: " . $e->getMessage());
        }
    }

    /**
     * Warm up totals for a customer order line.
     */
    public function warmupCustomerOrderLine(CustomerOrderLines $orderLine): void
    {
        try {
            $totals = $this->calculateCustomerOrderLineTotals($orderLine);
            
            $this->set('customer_order_line', $orderLine->id, 'prices', $totals);
            
            Log::info("Warmed up totals for customer order line {$orderLine->id}");
        } catch (\Exception $e) {
            Log::error("Failed to warm up totals for customer order line {$orderLine->id}: " . $e->getMessage());
        }
    }

    /**
     * Warm up totals for a customer order.
     */
    public function warmupCustomerOrder(CustomerOrders $order): void
    {
        try {
            $totals = $this->calculateCustomerOrderTotals($order);
            
            $this->set('customer_order', $order->id, 'prices', $totals);
            
            Log::info("Warmed up totals for customer order {$order->id}");
        } catch (\Exception $e) {
            Log::error("Failed to warm up totals for customer order {$order->id}: " . $e->getMessage());
        }
    }

    /**
     * Warm up totals for a shipment.
     */
    public function warmupShipment(Shipment $shipment): void
    {
        try {
            $totals = $this->calculateShipmentTotals($shipment);
            
            $this->set('shipment', $shipment->id, 'prices', $totals);
            
            Log::info("Warmed up totals for shipment {$shipment->id}");
        } catch (\Exception $e) {
            Log::error("Failed to warm up totals for shipment {$shipment->id}: " . $e->getMessage());
        }
    }

    /**
     * Calculate shipment line totals (replicates getTotalPricesAttribute logic).
     */
    private function calculateShipmentLineTotals(ShipmentLine $shipmentLine): array
    {
        $quote = 0;
        $quote_base = 0;
        $cmt = 0;
        $subtotal = 0;
        $subtotal_base = 0;
        $transport_budget = 0;
        $transport_currency = null;

        foreach ($shipmentLine->shipment_line_sizes as $qty) {
            $price = $shipmentLine->customer_order_lines->customer_order_line_quantities
                ->where('sizes_id', $qty->sizes_id)
                ->first()?->price_model;
            
            if (!empty($price)) {
                $pieces = empty($qty->shipped_qty) ? $qty->qty : $qty->shipped_qty;
                $quote += ($price['quote'] * $pieces);
                $quote_base += ($price['quote_base'] * $pieces);
                $cmt += ($price['cmt'] * $pieces);
                $subtotal += ($price['subtotal'] * $pieces);
                $subtotal_base += ($price['subtotal_base'] * $pieces);
                $transport_budget += ($price['transport_budget_base'] * $pieces);
                $transport_currency = $price['transport_currency'] ?? null;
            }
        }

        return [
            'quote' => $quote,
            'quote_base' => $quote_base,
            'cmt' => $cmt,
            'subtotal' => $subtotal,
            'subtotal_base' => $subtotal_base,
            'transport_budget' => $transport_budget,
            'transport_currency' => $transport_currency,
        ];
    }

    /**
     * Calculate customer order line totals.
     */
    private function calculateCustomerOrderLineTotals(CustomerOrderLines $orderLine): array
    {
        $quote = 0;
        $quote_base = 0;
        $cmt = 0;
        $subtotal = 0;
        $subtotal_base = 0;
        $transport_budget = 0;

        foreach ($orderLine->customer_order_line_quantities as $qty) {
            $price = $qty->price_model;
            if (!empty($price)) {
                $pieces = $qty->qty;
                $quote += ($price['quote'] * $pieces);
                $quote_base += ($price['quote_base'] * $pieces);
                $cmt += ($price['cmt'] * $pieces);
                $subtotal += ($price['subtotal'] * $pieces);
                $subtotal_base += ($price['subtotal_base'] * $pieces);
                $transport_budget += ($price['transport_budget_base'] * $pieces);
            }
        }

        return [
            'quote' => $quote,
            'quote_base' => $quote_base,
            'cmt' => $cmt,
            'subtotal' => $subtotal,
            'subtotal_base' => $subtotal_base,
            'transport_budget' => $transport_budget,
        ];
    }

    /**
     * Calculate customer order totals.
     */
    private function calculateCustomerOrderTotals(CustomerOrders $order): array
    {
        $total = 0;
        $total_base = 0;
        $total_qty = 0;

        foreach ($order->customer_order_lines as $line) {
            $lineTotals = $this->get('customer_order_line', $line->id, 'prices');
            if (!$lineTotals) {
                $lineTotals = $this->calculateCustomerOrderLineTotals($line);
                $this->set('customer_order_line', $line->id, 'prices', $lineTotals);
            }
            
            $total += $lineTotals['quote'] ?? 0;
            $total_base += $lineTotals['quote_base'] ?? 0;
            $total_qty += $line->customer_order_line_quantities->sum('qty');
        }

        return [
            'total_sale' => $total,
            'total_sale_base' => $total_base,
            'total_qty' => $total_qty,
        ];
    }

    /**
     * Calculate shipment totals.
     */
    private function calculateShipmentTotals(Shipment $shipment): array
    {
        $goods_value = 0;
        $transport_budget = 0;
        $goods_qty = 0;
        $cmt_total = 0;

        foreach ($shipment->shipment_lines as $sl) {
            $order = $sl->customer_order_lines->customer_orders;
            $isWholesale = $order->order_type === 'wholesale';

            foreach ($sl->customer_order_lines->customer_order_line_quantities as $colq) {
                $qty = $colq->qty;
                $goods_qty += $qty;
                $goods_value += ($isWholesale ? $colq->quote_base : $colq->price) * $qty;
                $transport_budget += ($isWholesale ? $colq->transport_budget : $colq->price) * $qty;
                if (isset($colq->cmt)) {
                    $cmt_total += $colq->cmt * $qty;
                }
            }
        }

        return [
            'goods_value' => $goods_value,
            'transport_budget' => $transport_budget,
            'goods_qty' => $goods_qty,
            'cmt_total' => $cmt_total,
        ];
    }

    /**
     * Get cache statistics.
     */
    public function getCacheStats(): array
    {
        $total = TotalCache::count();
        $fresh = TotalCache::fresh()->count();
        $stale = TotalCache::stale()->count();

        $byEntity = TotalCache::select('entity_type', DB::raw('count(*) as count'))
            ->groupBy('entity_type')
            ->pluck('count', 'entity_type')
            ->toArray();

        return [
            'total' => $total,
            'fresh' => $fresh,
            'stale' => $stale,
            'by_entity' => $byEntity,
        ];
    }

    /**
     * Clean up stale cache entries.
     */
    public function cleanupStale(int $daysOld = 30): int
    {
        $cutoff = now()->subDays($daysOld);
        
        return TotalCache::stale()
            ->where('updated_at', '<', $cutoff)
            ->delete();
    }
}