<?php

namespace App\Services;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Dompdf\Dompdf;
use Dompdf\Options;
use Exception;

class FileProcessingService
{
    public function processFile(UploadedFile $file, string $disk = 'private'): array
    {
        $extension = strtolower($file->getClientOriginalExtension());
        
        switch ($extension) {
            case 'eml':
                return $this->processEmlFile($file, $disk);
            case 'pdf':
                return $this->processPdfFile($file, $disk);
            default:
                throw new Exception("Unsupported file type: {$extension}");
        }
    }
    
    private function processEmlFile(UploadedFile $file, string $disk): array
    {
        // Read EML content
        $emlContent = file_get_contents($file->getRealPath());
        
        // Parse EML to extract content
        $parsed = $this->parseEml($emlContent);
        
        // Convert to PDF
        $pdfPath = $this->convertEmlToPdf($parsed, $disk);
        
        return [
            'type' => 'eml',
            'original_path' => $file->store('emails', $disk),
            'pdf_path' => $pdfPath,
            'subject' => $parsed['subject'] ?? 'No Subject',
            'from' => $parsed['from'] ?? 'Unknown',
            'date' => $parsed['date'] ?? now(),
            'content' => $parsed['content'] ?? '',
            'attachments' => $parsed['attachments'] ?? []
        ];
    }
    
    private function processPdfFile(UploadedFile $file, string $disk): array
    {
        // Store original PDF
        $pdfPath = $file->store('pdfs', $disk);
        
        // Extract text from PDF
        $text = $this->extractPdfText($file->getRealPath());
        
        // Extract images from PDF (if any)
        $images = $this->extractPdfImages($file->getRealPath(), $disk);
        
        return [
            'type' => 'pdf',
            'pdf_path' => $pdfPath,
            'text' => $text,
            'images' => $images,
            'page_count' => $this->getPdfPageCount($file->getRealPath())
        ];
    }
    
    private function parseEml(string $emlContent): array
    {
        $parsed = [
            'subject' => null,
            'from' => null,
            'date' => null,
            'content' => null,
            'attachments' => []
        ];
        
        // Parse headers
        $lines = explode("\n", $emlContent);
        $inHeaders = true;
        $content = '';
        
        foreach ($lines as $line) {
            if ($inHeaders) {
                if (empty(trim($line))) {
                    $inHeaders = false;
                    continue;
                }
                
                if (stripos($line, 'Subject:') === 0) {
                    $parsed['subject'] = trim(substr($line, 8));
                } elseif (stripos($line, 'From:') === 0) {
                    $parsed['from'] = trim(substr($line, 5));
                } elseif (stripos($line, 'Date:') === 0) {
                    $parsed['date'] = trim(substr($line, 5));
                }
            } else {
                $content .= $line . "\n";
            }
        }
        
        $parsed['content'] = $content;
        
        // Look for attachments in the content
        if (preg_match_all('/Content-Disposition: attachment[^;]*filename="([^"]+)"/i', $emlContent, $matches)) {
            $parsed['attachments'] = $matches[1];
        }
        
        return $parsed;
    }
    
    private function convertEmlToPdf(array $parsed, string $disk): string
    {
        $html = $this->generateHtmlFromEml($parsed);
        
        // Configure DomPDF
        $options = new Options();
        $options->set('defaultFont', 'Arial');
        $options->set('isRemoteEnabled', true);
        
        $dompdf = new Dompdf($options);
        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();
        
        // Save PDF
        $pdfPath = 'emails/' . uniqid() . '.pdf';
        Storage::disk($disk)->put($pdfPath, $dompdf->output());
        
        return $pdfPath;
    }
    
    private function generateHtmlFromEml(array $parsed): string
    {
        $html = '<html><head><meta charset="UTF-8"></head><body>';
        $html .= '<div style="font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px;">';
        
        // Header
        $html .= '<div style="border-bottom: 2px solid #ccc; padding-bottom: 10px; margin-bottom: 20px;">';
        $html .= '<h2 style="margin: 0; color: #333;">' . htmlspecialchars($parsed['subject'] ?? 'No Subject') . '</h2>';
        $html .= '<p style="margin: 5px 0; color: #666;">From: ' . htmlspecialchars($parsed['from'] ?? 'Unknown') . '</p>';
        $html .= '<p style="margin: 5px 0; color: #666;">Date: ' . htmlspecialchars($parsed['date'] ?? 'Unknown') . '</p>';
        $html .= '</div>';
        
        // Content
        $html .= '<div style="line-height: 1.6; color: #333;">';
        $html .= nl2br(htmlspecialchars($parsed['content'] ?? 'No content'));
        $html .= '</div>';
        
        // Attachments
        if (!empty($parsed['attachments'])) {
            $html .= '<div style="margin-top: 20px; padding-top: 20px; border-top: 1px solid #eee;">';
            $html .= '<h3 style="margin: 0 0 10px 0; color: #333;">Attachments:</h3>';
            $html .= '<ul style="margin: 0; padding-left: 20px;">';
            foreach ($parsed['attachments'] as $attachment) {
                $html .= '<li style="margin: 5px 0;">' . htmlspecialchars($attachment) . '</li>';
            }
            $html .= '</ul>';
            $html .= '</div>';
        }
        
        $html .= '</div></body></html>';
        
        return $html;
    }
    
    private function extractPdfText(string $pdfPath): string
    {
        try {
            // Use PhpSpreadsheet to read PDF (if it supports it)
            // For now, return a placeholder - you might need a dedicated PDF text extraction library
            return "PDF text extraction not implemented yet. File: " . basename($pdfPath);
        } catch (Exception $e) {
            return "Error extracting text: " . $e->getMessage();
        }
    }
    
    private function extractPdfImages(string $pdfPath, string $disk): array
    {
        // For now, return empty array
        // You would need a library like pdf2pic or similar to extract images
        return [];
    }
    
    private function getPdfPageCount(string $pdfPath): int
    {
        try {
            // Simple page count estimation
            $content = file_get_contents($pdfPath);
            return substr_count($content, '/Type/Page');
        } catch (Exception $e) {
            return 1;
        }
    }
}
