<?php

namespace App\Services;

use App\Models\Receipt;
use App\Services\Matching\ClassificationService;
use App\Services\VAT\UkVatService;
use App\Services\Matching\ReceiptMatchingService;
use App\Services\OCR\ReceiptOCRService;
use App\Services\Analysis\DiscountAnalysisService;
use Illuminate\Support\Facades\Log;

class OptimizedReceiptProcessingService
{
    public function __construct(
        private ReceiptOCRService $ocr,
        private ClassificationService $classification,
        private UkVatService $vat,
        private ReceiptMatchingService $matching,
        private DiscountAnalysisService $discountAnalysis,
    ) {}

    public function process(Receipt $receipt): void
    {
        $startTime = microtime(true);
        
        // First extract data from the receipt image using OCR
        $this->ocr->extractReceiptData($receipt);
        
        // Refresh the receipt to get the updated data and lines
        $receipt->refresh();
        
        // Then classify the receipt
        $this->classification->classifyReceipt($receipt);
        
        // Process lines with optimized batching
        $this->processLinesOptimized($receipt);
        
        // Analyze discounts and link them to line items
        $this->discountAnalysis->analyzeDiscounts($receipt);
        
        // Finally attempt to match the receipt to statement transactions
        $this->matching->attemptMatchForReceipt($receipt);
        
        $endTime = microtime(true);
        $processingTime = round($endTime - $startTime, 2);
        
        Log::info("Receipt processing completed", [
            'receipt_id' => $receipt->id,
            'lines_count' => $receipt->lines->count(),
            'processing_time' => $processingTime . 's'
        ]);
    }

    /**
     * Process receipt lines with optimized batching and reduced API calls
     */
    private function processLinesOptimized(Receipt $receipt): void
    {
        if ($receipt->lines->isEmpty()) {
            return;
        }

        $lines = $receipt->lines;
        $batchSize = 2; // Process 2 lines at a time to balance speed vs API limits
        
        // Process lines in small batches
        $batches = $lines->chunk($batchSize);
        
        foreach ($batches as $batch) {
            $this->processBatch($batch);
        }
    }

    private function processBatch($lines): void
    {
        foreach ($lines as $line) {
            // Process classification and VAT analysis
            $this->classification->classifyReceiptLine($line);
            $this->vat->analyzeLine($line);
        }
    }
}
