#!/usr/bin/env node

const { Pool } = require('pg');
const bcrypt = require('bcrypt');
const fs = require('fs');
const readline = require('readline');

require('dotenv').config();

// Database configuration
const dbConfig = process.env.INSTANCE_UNIX_SOCKET
  ? {
      user: process.env.DB_USER || 'postgres',
      password: process.env.DB_PASSWORD,
      database: process.env.DB_NAME || 'photowebsite',
      host: process.env.INSTANCE_UNIX_SOCKET,
    }
  : {
      user: process.env.DB_USER || 'postgres',
      password: process.env.DB_PASSWORD,
      database: process.env.DB_NAME || 'photowebsite',
      host: process.env.DB_HOST || 'localhost',
      port: parseInt(process.env.DB_PORT || '5432', 10),
    };

const pool = new Pool(dbConfig);

const rl = readline.createInterface({
  input: process.stdin,
  output: process.stdout
});

function question(prompt) {
  return new Promise((resolve) => {
    rl.question(prompt, resolve);
  });
}

function questionPassword(prompt) {
  return new Promise((resolve) => {
    const stdin = process.stdin;
    stdin.resume();
    stdin.setRawMode(true);
    stdin.setEncoding('utf8');
    
    process.stdout.write(prompt);
    let password = '';
    
    stdin.on('data', function onData(char) {
      if (char === '\n' || char === '\r' || char === '\u0004') {
        stdin.setRawMode(false);
        stdin.removeListener('data', onData);
        process.stdout.write('\n');
        resolve(password);
      } else if (char === '\u0003') {
        process.exit();
      } else if (char === '\u007f') {
        if (password.length > 0) {
          password = password.slice(0, -1);
          process.stdout.write('\b \b');
        }
      } else {
        password += char;
        process.stdout.write('*');
      }
    });
  });
}

async function runMigration() {
  console.log('Step 1: Running superuser migration...\n');
  
  const migrationPath = './migrations/002-add-superuser-support.sql';
  
  if (!fs.existsSync(migrationPath)) {
    console.log('Warning: Migration file not found at', migrationPath);
    console.log('The database may already have superuser support\n');
    return;
  }
  
  const sql = fs.readFileSync(migrationPath, 'utf8');
  
  try {
    await pool.query(sql);
    console.log('✓ Migration completed successfully\n');
  } catch (error) {
    if (error.message.includes('already exists')) {
      console.log('✓ Tables already exist, skipping migration\n');
    } else {
      console.error('✗ Migration failed:', error.message);
      throw error;
    }
  }
}

async function createSuperuser() {
  console.log('Step 2: Creating superuser account...\n');
  
  const username = await question('Enter username (lowercase, alphanumeric and hyphens only): ');
  const email = await question('Enter email address: ');
  const password = await questionPassword('Enter password: ');
  const passwordConfirm = await questionPassword('Confirm password: ');
  
  console.log('');
  
  if (password !== passwordConfirm) {
    throw new Error('Passwords do not match');
  }
  
  if (!username.match(/^[a-z0-9-]{3,50}$/)) {
    throw new Error('Username must be 3-50 characters, lowercase letters, numbers, and hyphens only');
  }
  
  if (!email.match(/^[^\s@]+@[^\s@]+\.[^\s@]+$/)) {
    throw new Error('Invalid email format');
  }
  
  if (password.length < 8) {
    throw new Error('Password must be at least 8 characters long');
  }
  
  console.log('Generating password hash...');
  const passwordHash = await bcrypt.hash(password, 10);
  
  console.log('Creating superuser account...');
  
  try {
    const result = await pool.query(
      `INSERT INTO users (
        username, 
        email, 
        password_hash, 
        full_name,
        subscription_tier, 
        subscription_status,
        is_superuser,
        max_photos,
        max_storage_bytes
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
      RETURNING id, username, email, is_superuser`,
      [
        username.toLowerCase().trim(),
        email.toLowerCase().trim(),
        passwordHash,
        'Superuser Administrator',
        'business',
        'active',
        true,
        999999,
        999999999999
      ]
    );
    
    console.log('\n===================================');
    console.log('✓ Superuser created successfully!');
    console.log('===================================\n');
    console.log('User ID:', result.rows[0].id);
    console.log('Username:', result.rows[0].username);
    console.log('Email:', result.rows[0].email);
    console.log('Superuser:', result.rows[0].is_superuser);
    
  } catch (error) {
    if (error.message.includes('duplicate key') || error.message.includes('already exists')) {
      console.log('\nUser already exists. Trying to update existing user to superuser...\n');
      
      const result = await pool.query(
        `UPDATE users 
         SET is_superuser = TRUE,
             subscription_status = 'active',
             subscription_tier = 'business'
         WHERE username = $1 OR email = $2
         RETURNING id, username, email, is_superuser`,
        [username.toLowerCase().trim(), email.toLowerCase().trim()]
      );
      
      if (result.rows.length > 0) {
        console.log('✓ Existing user updated to superuser status');
        console.log('User ID:', result.rows[0].id);
        console.log('Username:', result.rows[0].username);
        console.log('Email:', result.rows[0].email);
      } else {
        throw new Error('Failed to update user');
      }
    } else {
      throw error;
    }
  }
  
  console.log('\nYou can now log in at: http://app.localhost:3000/login');
  console.log('After logging in, access the superuser portal at:');
  console.log('http://app.localhost:3000/superuser\n');
}

async function main() {
  console.log('===================================');
  console.log('Vybe Photo - Superuser Setup');
  console.log('===================================\n');
  
  console.log('Database Configuration:');
  console.log('  Host:', dbConfig.host);
  console.log('  Port:', dbConfig.port);
  console.log('  Database:', dbConfig.database);
  console.log('  User:', dbConfig.user);
  console.log('');
  
  try {
    // Test connection
    await pool.query('SELECT 1');
    console.log('✓ Database connection successful\n');
    
    await runMigration();
    await createSuperuser();
    
    console.log('Setup complete!');
    
  } catch (error) {
    console.error('\n✗ Error:', error.message);
    process.exit(1);
  } finally {
    await pool.end();
    rl.close();
  }
}

main();

