<?php

namespace App\Services;

use App\Contracts\LibraryService as LibraryServiceContract;
use App\Models\Tag;
use App\Models\Track;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Storage;

class LibraryServiceImpl implements LibraryServiceContract
{
    public function listTracks(?array $tagSlugs, ?string $level, int $perPage = 50, ?string $search = null): LengthAwarePaginator
    {
        $query = Track::query()
            ->with(['voiceActor:id,name', 'tags:id,slug'])
            ->when($level, fn ($q) => $q->where('level', $level))
            ->when($search, function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->where('title', 'like', "%{$search}%")
                       ->orWhere('description', 'like', "%{$search}%");
                });
            })
            ->when($tagSlugs, function ($q) use ($tagSlugs) {
                $q->whereHas('tags', function ($tq) use ($tagSlugs) {
                    $tq->whereIn('slug', $tagSlugs);
                });
            })
            ->whereNotNull('published_at')
            ->orderByDesc('published_at');

        return $query->paginate($perPage);
    }

    public function signMediaUrl(?string $objectKey, int $ttlSeconds = 300): ?string
    {
        if (!$objectKey) {
            return null;
        }
        // Use dedicated MediaUrlService for signing/fallback
        return app(\App\Contracts\MediaUrlService::class)->signed($objectKey, null, $ttlSeconds);
    }

    public function getSignedAudioUrl(Track $track, int $ttlSeconds = 300): ?string
    {
        return $this->signMediaUrl($track->audio_url, $ttlSeconds);
    }

    public function getSignedImageUrl(Track $track, int $ttlSeconds = 300): ?string
    {
        return $this->signMediaUrl($track->image_url, $ttlSeconds);
    }
}


