<?php

namespace App\Console\Commands;

use App\Models\StatementTransaction;
use App\Services\AI\ExpenseClassificationAgent;
use Illuminate\Console\Command;

class TestExpenseClassification extends Command
{
    protected $signature = 'expense:test-classification {transaction_id?}';
    protected $description = 'Test the expense classification agent on a statement transaction';

    public function handle(ExpenseClassificationAgent $agent): int
    {
        $transactionId = $this->argument('transaction_id');
        
        if ($transactionId) {
            $transaction = StatementTransaction::find($transactionId);
            if (!$transaction) {
                $this->error("Transaction with ID {$transactionId} not found.");
                return 1;
            }
        } else {
            // Get a random unclassified transaction
            $transaction = StatementTransaction::whereNull('department_id')
                ->orWhereNull('account_id')
                ->first();
                
            if (!$transaction) {
                $this->info('No unclassified transactions found. Using a random transaction...');
                $transaction = StatementTransaction::inRandomOrder()->first();
            }
        }

        if (!$transaction) {
            $this->error('No transactions found.');
            return 1;
        }

        $this->info("Testing classification for transaction ID: {$transaction->id}");
        $this->info("Merchant: {$transaction->merchant_name}");
        $this->info("Amount: {$transaction->amount} {$transaction->currency}");
        $this->info("Date: {$transaction->transaction_date}");

        $result = $agent->classifyTransaction($transaction);

        $this->info("\nClassification Result:");
        $this->info("Success: " . ($result['success'] ? 'Yes' : 'No'));
        $this->info("Confidence: " . ($result['confidence'] ?? 'N/A'));
        $this->info("Reasoning: " . ($result['reasoning'] ?? 'N/A'));
        
        if (!empty($result['updates'])) {
            $this->info("\nUpdates Applied:");
            foreach ($result['updates'] as $field => $value) {
                $this->info("- {$field}: {$value}");
            }
        }

        if (!empty($result['split_suggestion'])) {
            $this->info("\nSplit Suggestion:");
            $this->info(json_encode($result['split_suggestion'], JSON_PRETTY_PRINT));
        }

        return 0;
    }
}

