<?php

namespace App\Jobs;

use App\Models\Receipt;
use App\Services\ReceiptMergingService;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class MergeReceiptsJob implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $timeout = 300; // 5 minutes for merging
    public int $maxExceptions = 2;

    public function __construct(
        public array $receiptIds,
        public int $userId
    ) {}

    public function backoff(): array
    {
        return [30, 120, 300];
    }

    public function retryUntil(): \DateTimeInterface
    {
        return now()->addHours(2);
    }

    public function handle(ReceiptMergingService $mergingService): void
    {
        try {
            Log::info('Starting receipt merge job', [
                'receipt_ids' => $this->receiptIds,
                'user_id' => $this->userId
            ]);

            // Validate receipts exist and belong to user
            $receipts = Receipt::whereIn('id', $this->receiptIds)
                ->where('user_id', $this->userId)
                ->get();

            if ($receipts->count() !== count($this->receiptIds)) {
                throw new \Exception('Some receipts not found or do not belong to user');
            }

            // Perform the merge
            $mergedReceipt = $mergingService->mergeReceipts($this->receiptIds);

            // Dispatch job to process the merged receipt
            ProcessMergedReceipt::dispatch($mergedReceipt->id);

            Log::info('Receipt merge job completed successfully', [
                'merged_receipt_id' => $mergedReceipt->id,
                'original_receipt_ids' => $this->receiptIds,
                'user_id' => $this->userId
            ]);

        } catch (\Throwable $e) {
            Log::error('Receipt merge job failed', [
                'receipt_ids' => $this->receiptIds,
                'user_id' => $this->userId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // Re-throw to trigger retry mechanism
            throw $e;
        }
    }

    public function failed(\Throwable $exception): void
    {
        Log::error('Receipt merge job failed permanently', [
            'receipt_ids' => $this->receiptIds,
            'user_id' => $this->userId,
            'error' => $exception->getMessage()
        ]);
    }
}
