<?php

namespace App\Services;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\Facades\Image;

class ImageResizeService
{
    private int $maxWidth;
    private int $maxHeight;
    private int $quality;

    public function __construct()
    {
        $this->maxWidth = config('receipts.max_width', 1024);
        $this->maxHeight = config('receipts.max_height', 1024);
        $this->quality = config('receipts.quality', 85);
    }

    /**
     * Resize and optimize an uploaded image file
     */
    public function resizeAndStore(UploadedFile $file, string $path, string $disk = 'local'): string
    {
        // Create the directory if it doesn't exist
        $directory = dirname($path);
        if (!Storage::disk($disk)->exists($directory)) {
            Storage::disk($disk)->makeDirectory($directory);
        }

        // Get the full path for the resized image
        $fullPath = Storage::disk($disk)->path($path);

        // Load the image using GD (built into PHP)
        $image = $this->loadImage($file);
        if (!$image) {
            throw new \Exception('Could not load image file');
        }

        // Get original dimensions
        $originalWidth = imagesx($image);
        $originalHeight = imagesy($image);

        // Calculate new dimensions maintaining aspect ratio
        $newDimensions = $this->calculateNewDimensions($originalWidth, $originalHeight);

        // Resize the image if needed
        if ($newDimensions['width'] !== $originalWidth || $newDimensions['height'] !== $originalHeight) {
            $resizedImage = imagecreatetruecolor($newDimensions['width'], $newDimensions['height']);
            
            // Preserve transparency for PNG images
            if ($file->getMimeType() === 'image/png') {
                imagealphablending($resizedImage, false);
                imagesavealpha($resizedImage, true);
                $transparent = imagecolorallocatealpha($resizedImage, 255, 255, 255, 127);
                imagefill($resizedImage, 0, 0, $transparent);
            }

            imagecopyresampled(
                $resizedImage, $image,
                0, 0, 0, 0,
                $newDimensions['width'], $newDimensions['height'],
                $originalWidth, $originalHeight
            );

            imagedestroy($image);
            $image = $resizedImage;
        }

        // Save the resized image
        $this->saveImage($image, $fullPath, $file->getMimeType());
        imagedestroy($image);

        return $path;
    }

    /**
     * Load an image from an uploaded file using GD
     */
    private function loadImage(UploadedFile $file)
    {
        $mimeType = $file->getMimeType();
        $tempPath = $file->getRealPath();

        switch ($mimeType) {
            case 'image/jpeg':
                return imagecreatefromjpeg($tempPath);
            case 'image/png':
                return imagecreatefrompng($tempPath);
            case 'image/gif':
                return imagecreatefromgif($tempPath);
            case 'image/webp':
                return imagecreatefromwebp($tempPath);
            default:
                return false;
        }
    }

    /**
     * Calculate new dimensions maintaining aspect ratio
     */
    private function calculateNewDimensions(int $originalWidth, int $originalHeight): array
    {
        // If image is already smaller than max dimensions, keep original size
        if ($originalWidth <= $this->maxWidth && $originalHeight <= $this->maxHeight) {
            return [
                'width' => $originalWidth,
                'height' => $originalHeight
            ];
        }

        // Calculate scaling factor
        $widthRatio = $this->maxWidth / $originalWidth;
        $heightRatio = $this->maxHeight / $originalHeight;
        $scale = min($widthRatio, $heightRatio);

        return [
            'width' => (int) round($originalWidth * $scale),
            'height' => (int) round($originalHeight * $scale)
        ];
    }

    /**
     * Save image to file with appropriate format and quality
     */
    private function saveImage($image, string $path, string $mimeType): void
    {
        switch ($mimeType) {
            case 'image/jpeg':
                imagejpeg($image, $path, $this->quality);
                break;
            case 'image/png':
                // For PNG, we don't use quality parameter as it's lossless
                imagepng($image, $path, 9);
                break;
            case 'image/gif':
                imagegif($image, $path);
                break;
            case 'image/webp':
                imagewebp($image, $path, $this->quality);
                break;
            default:
                // Default to JPEG if format is not supported
                imagejpeg($image, $path, $this->quality);
        }
    }

    /**
     * Get file size after resizing (estimate)
     */
    public function getEstimatedSize(int $originalWidth, int $originalHeight, string $mimeType): int
    {
        $newDimensions = $this->calculateNewDimensions($originalWidth, $originalHeight);
        $pixels = $newDimensions['width'] * $newDimensions['height'];
        
        // Rough estimate based on format and quality
        switch ($mimeType) {
            case 'image/jpeg':
                return (int) ($pixels * 0.5 * ($this->quality / 100)); // Rough estimate
            case 'image/png':
                return (int) ($pixels * 1.5); // PNG is typically larger
            case 'image/webp':
                return (int) ($pixels * 0.3 * ($this->quality / 100));
            default:
                return (int) ($pixels * 0.5);
        }
    }
}
