<?php

namespace App\Services;

use App\Jobs\ProcessReceipt;
use App\Jobs\ProcessReceiptOCR;
use App\Jobs\ProcessReceiptDeduplication;
use App\Jobs\ProcessReceiptMatching;
use App\Jobs\MatchReceipt;
use App\Models\Receipt;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Log;

class ParallelProcessingService
{
    /**
     * Process multiple receipts with the correct 2-stage workflow:
     * 1. OCR ALL receipts first
     * 2. THEN match to statements
     */
    public function processReceiptsOptimized(array $receiptIds): void
    {
        Log::info("Starting 2-stage receipt processing", ['receipt_count' => count($receiptIds)]);
        
        // Stage 1: OCR ALL receipts first
        $this->processOCRStage($receiptIds);
    }
    
    /**
     * Stage 1: Process OCR for all receipts in controlled parallel batches.
     * Dispatches one batch at a time to cap in-flight OCR jobs to the batch size.
     */
    private function processOCRStage(array $receiptIds): void
    {
        $batchSize = (int) config('receipts.ocr_batch_size', 5);
        $totalBatches = (int) ceil(max(count($receiptIds), 1) / max($batchSize, 1));

        Log::info("Processing OCR in batches", [
            'batch_size' => $batchSize,
            'total_batches' => $totalBatches,
            'receipt_count' => count($receiptIds)
        ]);

        // Track orchestration state in cache
        $cacheKey = 'ocr_processing_' . uniqid();
        \Cache::put($cacheKey, [
            'receipt_ids' => array_values($receiptIds),
            'batch_size' => $batchSize,
            'total_batches' => $totalBatches,
            'next_index' => 0,
        ], now()->addHours(2));

        // Kick off the first batch; subsequent batches are dispatched on completion
        $this->dispatchNextOcrBatch($cacheKey);
    }

    /**
     * Dispatch the next OCR batch if any remain.
     */
    private function dispatchNextOcrBatch(string $cacheKey): void
    {
        $data = \Cache::get($cacheKey);
        if (!$data) {
            Log::warning("OCR dispatch state missing", ['cache_key' => $cacheKey]);
            return;
        }

        $receiptIds = (array) ($data['receipt_ids'] ?? []);
        $batchSize = (int) ($data['batch_size'] ?? 5);
        $totalBatches = (int) ($data['total_batches'] ?? 0);
        $index = (int) ($data['next_index'] ?? 0);

        if ($index >= $totalBatches) {
            // Already finished
            return;
        }

        $offset = $index * $batchSize;
        $batch = array_slice($receiptIds, $offset, $batchSize);

        $jobs = [];
        foreach ($batch as $receiptId) {
            $jobs[] = new ProcessReceiptOCR($receiptId, null, false);
        }

        Bus::batch($jobs)
            ->name("OCR Stage - Batch " . ($index + 1))
            ->allowFailures()
            ->onQueue('ocr')
            ->then(function () use ($cacheKey) {
                $this->handleOCRBatchCompletion($cacheKey);
            })
            ->catch(function (\Throwable $e) use ($cacheKey) {
                Log::error("OCR batch failed", ['error' => $e->getMessage()]);
                $this->handleOCRBatchCompletion($cacheKey);
            })
            ->dispatch();

        Log::info("Dispatched OCR batch", [
            'batch' => $index + 1,
            'cache_key' => $cacheKey,
            'count' => count($batch),
        ]);
    }
    
    /**
     * Handle OCR batch completion and trigger deduplication when all batches are done
     */
    private function handleOCRBatchCompletion(string $cacheKey): void
    {
        $data = \Cache::get($cacheKey);
        if (!$data) {
            Log::warning("OCR completion data not found in cache", ['cache_key' => $cacheKey]);
            return;
        }

        $data['next_index'] = (int) ($data['next_index'] ?? 0) + 1;
        $nextIndex = $data['next_index'];
        $totalBatches = (int) ($data['total_batches'] ?? 0);

        Log::info("OCR batch completed", [
            'completed_batch' => $nextIndex,
            'total_batches' => $totalBatches,
        ]);

        if ($nextIndex >= $totalBatches) {
            Log::info("All OCR batches completed, starting matching stage", [
                'receipt_count' => count((array) ($data['receipt_ids'] ?? []))
            ]);

            $receiptIds = (array) ($data['receipt_ids'] ?? []);
            \Cache::forget($cacheKey);
            $this->processMatchingStage($receiptIds);
            return;
        }

        // Persist state and dispatch the next batch
        \Cache::put($cacheKey, $data, now()->addHours(2));
        $this->dispatchNextOcrBatch($cacheKey);
    }
    
    
    /**
     * Stage 2: Process matching for all receipts
     */
    private function processMatchingStage(array $receiptIds): void
    {
        $batches = array_chunk($receiptIds, 10); // Larger batches for matching
        $totalBatches = count($batches);
        
        Log::info("Processing matching in batches", ['total_batches' => $totalBatches, 'receipt_count' => count($receiptIds)]);
        
        // Store the receipt IDs in cache to track completion
        $cacheKey = 'matching_processing_' . uniqid();
        \Cache::put($cacheKey, [
            'receipt_ids' => $receiptIds,
            'total_batches' => $totalBatches,
            'completed_batches' => 0
        ], now()->addHours(2));
        
        foreach ($batches as $batchIndex => $batch) {
            $jobs = [];
            
            // Create matching jobs
            foreach ($batch as $receiptId) {
                $jobs[] = new ProcessReceiptMatching($receiptId);
            }
            
            // Dispatch batch with completion callback
            Bus::batch($jobs)
                ->name("Matching Stage - Batch " . ($batchIndex + 1))
                ->allowFailures()
                ->onQueue('matching')
                ->then(function () use ($cacheKey, $batchIndex, $totalBatches) {
                    $this->handleMatchingBatchCompletion($cacheKey, $batchIndex + 1, $totalBatches);
                })
                ->catch(function (\Throwable $e) use ($cacheKey, $batchIndex) {
                    Log::error("Matching batch failed", ['batch' => $batchIndex + 1, 'error' => $e->getMessage()]);
                    $this->handleMatchingBatchCompletion($cacheKey, $batchIndex + 1, $totalBatches);
                })
                ->dispatch();
                
            Log::info("Dispatched matching batch", ['batch' => $batchIndex + 1, 'receipt_ids' => $batch]);
        }
    }
    
    /**
     * Handle matching batch completion and log final completion
     */
    private function handleMatchingBatchCompletion(string $cacheKey, int $completedBatch, int $totalBatches): void
    {
        $data = \Cache::get($cacheKey);
        if (!$data) {
            Log::warning("Matching completion data not found in cache", ['cache_key' => $cacheKey]);
            return;
        }
        
        $data['completed_batches']++;
        \Cache::put($cacheKey, $data, now()->addHours(2));
        
        Log::info("Matching batch completed", [
            'completed_batch' => $completedBatch,
            'total_batches' => $totalBatches,
            'completed_count' => $data['completed_batches']
        ]);
        
        // If all batches are complete, log final completion
        if ($data['completed_batches'] >= $totalBatches) {
            Log::info("All matching batches completed - 2-stage receipt processing finished", [
                'receipt_count' => count($data['receipt_ids'])
            ]);
            
            // Clean up cache
            \Cache::forget($cacheKey);
        }
    }
    
    /**
     * Legacy method for backward compatibility - now uses the new 2-stage workflow
     */
    public function processReceiptsInParallel(array $receiptIds, int $batchSize = 5): void
    {
        $this->processReceiptsOptimized($receiptIds);
    }
    
    /**
     * Legacy method for backward compatibility - now uses the new 2-stage workflow
     */
    public function processMatchingInParallel(array $receiptIds, int $batchSize = 10): void
    {
        $this->processMatchingStage($receiptIds);
    }
}



