<?php

namespace App\Services;

use App\Jobs\ProcessReceipt;
use App\Jobs\ProcessReceiptOCR;
use App\Jobs\ProcessReceiptDeduplication;
use App\Jobs\ProcessReceiptMatching;
use App\Jobs\MatchReceipt;
use App\Models\Receipt;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Log;

class ParallelProcessingService
{
    /**
     * Process multiple receipts with the correct 3-stage workflow:
     * 1. OCR ALL receipts first
     * 2. THEN check for duplicates/grouping
     * 3. THEN match to statements
     */
    public function processReceiptsOptimized(array $receiptIds): void
    {
        Log::info("Starting 3-stage receipt processing", ['receipt_count' => count($receiptIds)]);
        
        // Stage 1: OCR ALL receipts first
        $this->processOCRStage($receiptIds);
    }
    
    /**
     * Stage 1: Process OCR for all receipts in parallel batches
     */
    private function processOCRStage(array $receiptIds): void
    {
        $batches = array_chunk($receiptIds, 3); // Smaller batches for better API rate limiting
        $totalBatches = count($batches);
        
        Log::info("Processing OCR in batches", ['total_batches' => $totalBatches, 'receipt_count' => count($receiptIds)]);
        
        // Store the receipt IDs in cache to track completion
        $cacheKey = 'ocr_processing_' . uniqid();
        \Cache::put($cacheKey, [
            'receipt_ids' => $receiptIds,
            'total_batches' => $totalBatches,
            'completed_batches' => 0
        ], now()->addHours(2));
        
        foreach ($batches as $batchIndex => $batch) {
            $jobs = [];
            
            // Create OCR processing jobs
            foreach ($batch as $receiptId) {
                $jobs[] = new ProcessReceiptOCR($receiptId);
            }
            
            // Dispatch batch with completion callback
            Bus::batch($jobs)
                ->name("OCR Stage - Batch " . ($batchIndex + 1))
                ->allowFailures()
                ->onQueue('ocr')
                ->then(function () use ($cacheKey, $batchIndex, $totalBatches) {
                    $this->handleOCRBatchCompletion($cacheKey, $batchIndex + 1, $totalBatches);
                })
                ->catch(function (\Throwable $e) use ($cacheKey, $batchIndex) {
                    Log::error("OCR batch failed", ['batch' => $batchIndex + 1, 'error' => $e->getMessage()]);
                    $this->handleOCRBatchCompletion($cacheKey, $batchIndex + 1, $totalBatches);
                })
                ->dispatch();
                
            Log::info("Dispatched OCR batch", ['batch' => $batchIndex + 1, 'receipt_ids' => $batch]);
        }
    }
    
    /**
     * Handle OCR batch completion and trigger deduplication when all batches are done
     */
    private function handleOCRBatchCompletion(string $cacheKey, int $completedBatch, int $totalBatches): void
    {
        $data = \Cache::get($cacheKey);
        if (!$data) {
            Log::warning("OCR completion data not found in cache", ['cache_key' => $cacheKey]);
            return;
        }
        
        $data['completed_batches']++;
        \Cache::put($cacheKey, $data, now()->addHours(2));
        
        Log::info("OCR batch completed", [
            'completed_batch' => $completedBatch,
            'total_batches' => $totalBatches,
            'completed_count' => $data['completed_batches']
        ]);
        
        // If all batches are complete, start deduplication
        if ($data['completed_batches'] >= $totalBatches) {
            Log::info("All OCR batches completed, starting deduplication stage", [
                'receipt_count' => count($data['receipt_ids'])
            ]);
            
            // Clean up cache
            \Cache::forget($cacheKey);
            
            // Start deduplication stage
            $this->processDeduplicationStage($data['receipt_ids']);
        }
    }
    
    /**
     * Schedule deduplication stage to run after OCR completes
     */
    private function scheduleDeduplicationStage(array $receiptIds): void
    {
        // This will be called by the OCR batch completion callback
        // No immediate action needed here
    }
    
    /**
     * Stage 2: Process deduplication for all receipts
     */
    private function processDeduplicationStage(array $receiptIds): void
    {
        $batches = array_chunk($receiptIds, 5); // Slightly larger batches for deduplication
        
        foreach ($batches as $batchIndex => $batch) {
            $jobs = [];
            
            // Create deduplication jobs
            foreach ($batch as $receiptId) {
                $jobs[] = new ProcessReceiptDeduplication($receiptId);
            }
            
            // Dispatch batch
            Bus::batch($jobs)
                ->name("Deduplication Stage - Batch " . ($batchIndex + 1))
                ->allowFailures()
                ->onQueue('deduplication')
                ->then(function () use ($receiptIds) {
                    Log::info("Deduplication stage completed, scheduling matching", ['receipt_count' => count($receiptIds)]);
                    $this->processMatchingStage($receiptIds);
                })
                ->dispatch();
                
            Log::info("Dispatched deduplication batch", ['batch' => $batchIndex + 1, 'receipt_ids' => $batch]);
        }
    }
    
    /**
     * Schedule matching stage to run after deduplication completes
     */
    private function scheduleMatchingStage(array $receiptIds): void
    {
        // This will be called by the deduplication batch completion callback
        // No immediate action needed here
    }
    
    /**
     * Stage 3: Process matching for all receipts
     */
    private function processMatchingStage(array $receiptIds): void
    {
        $batches = array_chunk($receiptIds, 10); // Larger batches for matching
        
        foreach ($batches as $batchIndex => $batch) {
            $jobs = [];
            
            // Create matching jobs
            foreach ($batch as $receiptId) {
                $jobs[] = new ProcessReceiptMatching($receiptId);
            }
            
            // Dispatch batch
            Bus::batch($jobs)
                ->name("Matching Stage - Batch " . ($batchIndex + 1))
                ->allowFailures()
                ->onQueue('matching')
                ->then(function () use ($receiptIds) {
                    Log::info("Matching stage completed - all receipt processing finished", ['receipt_count' => count($receiptIds)]);
                })
                ->dispatch();
                
            Log::info("Dispatched matching batch", ['batch' => $batchIndex + 1, 'receipt_ids' => $batch]);
        }
    }
    
    /**
     * Legacy method for backward compatibility - now uses the new 3-stage workflow
     */
    public function processReceiptsInParallel(array $receiptIds, int $batchSize = 5): void
    {
        $this->processReceiptsOptimized($receiptIds);
    }
    
    /**
     * Legacy method for backward compatibility - now uses the new 3-stage workflow
     */
    public function processMatchingInParallel(array $receiptIds, int $batchSize = 10): void
    {
        $this->processMatchingStage($receiptIds);
    }
}



