<?php

namespace App\Services;

use App\Jobs\ProcessReceipt;
use App\Jobs\ProcessReceiptOCR;
use App\Jobs\ProcessReceiptDeduplication;
use App\Jobs\ProcessReceiptMatching;
use App\Jobs\MatchReceipt;
use App\Models\Receipt;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Log;

class ParallelProcessingService
{
    /**
     * Process multiple receipts with the correct 2-stage workflow:
     * 1. OCR ALL receipts first
     * 2. THEN match to statements
     */
    public function processReceiptsOptimized(array $receiptIds): void
    {
        Log::info("Starting 2-stage receipt processing", ['receipt_count' => count($receiptIds)]);
        
        // Stage 1: OCR ALL receipts first
        $this->processOCRStage($receiptIds);
    }
    
    /**
     * Stage 1: Process OCR for all receipts in parallel batches
     */
    private function processOCRStage(array $receiptIds): void
    {
        $batches = array_chunk($receiptIds, 3); // Smaller batches for better API rate limiting
        $totalBatches = count($batches);
        
        Log::info("Processing OCR in batches", ['total_batches' => $totalBatches, 'receipt_count' => count($receiptIds)]);
        
        // Store the receipt IDs in cache to track completion
        $cacheKey = 'ocr_processing_' . uniqid();
        \Cache::put($cacheKey, [
            'receipt_ids' => $receiptIds,
            'total_batches' => $totalBatches,
            'completed_batches' => 0
        ], now()->addHours(2));
        
        foreach ($batches as $batchIndex => $batch) {
            $jobs = [];
            
            // Create OCR processing jobs
            foreach ($batch as $receiptId) {
                // Disable auto-matching during the OCR stage when orchestrated in batches
                $jobs[] = new ProcessReceiptOCR($receiptId, null, false);
            }
            
            // Dispatch batch with completion callback
            Bus::batch($jobs)
                ->name("OCR Stage - Batch " . ($batchIndex + 1))
                ->allowFailures()
                ->onQueue('ocr')
                ->then(function () use ($cacheKey, $batchIndex, $totalBatches) {
                    $this->handleOCRBatchCompletion($cacheKey, $batchIndex + 1, $totalBatches);
                })
                ->catch(function (\Throwable $e) use ($cacheKey, $batchIndex) {
                    Log::error("OCR batch failed", ['batch' => $batchIndex + 1, 'error' => $e->getMessage()]);
                    $this->handleOCRBatchCompletion($cacheKey, $batchIndex + 1, $totalBatches);
                })
                ->dispatch();
                
            Log::info("Dispatched OCR batch", ['batch' => $batchIndex + 1, 'receipt_ids' => $batch]);
        }
    }
    
    /**
     * Handle OCR batch completion and trigger deduplication when all batches are done
     */
    private function handleOCRBatchCompletion(string $cacheKey, int $completedBatch, int $totalBatches): void
    {
        $data = \Cache::get($cacheKey);
        if (!$data) {
            Log::warning("OCR completion data not found in cache", ['cache_key' => $cacheKey]);
            return;
        }
        
        $data['completed_batches']++;
        \Cache::put($cacheKey, $data, now()->addHours(2));
        
        Log::info("OCR batch completed", [
            'completed_batch' => $completedBatch,
            'total_batches' => $totalBatches,
            'completed_count' => $data['completed_batches']
        ]);
        
        // If all batches are complete, start matching
        if ($data['completed_batches'] >= $totalBatches) {
            Log::info("All OCR batches completed, starting matching stage", [
                'receipt_count' => count($data['receipt_ids'])
            ]);
            
            // Clean up cache
            \Cache::forget($cacheKey);
            
            // Start matching stage
            $this->processMatchingStage($data['receipt_ids']);
        }
    }
    
    
    /**
     * Stage 2: Process matching for all receipts
     */
    private function processMatchingStage(array $receiptIds): void
    {
        $batches = array_chunk($receiptIds, 10); // Larger batches for matching
        $totalBatches = count($batches);
        
        Log::info("Processing matching in batches", ['total_batches' => $totalBatches, 'receipt_count' => count($receiptIds)]);
        
        // Store the receipt IDs in cache to track completion
        $cacheKey = 'matching_processing_' . uniqid();
        \Cache::put($cacheKey, [
            'receipt_ids' => $receiptIds,
            'total_batches' => $totalBatches,
            'completed_batches' => 0
        ], now()->addHours(2));
        
        foreach ($batches as $batchIndex => $batch) {
            $jobs = [];
            
            // Create matching jobs
            foreach ($batch as $receiptId) {
                $jobs[] = new ProcessReceiptMatching($receiptId);
            }
            
            // Dispatch batch with completion callback
            Bus::batch($jobs)
                ->name("Matching Stage - Batch " . ($batchIndex + 1))
                ->allowFailures()
                ->onQueue('matching')
                ->then(function () use ($cacheKey, $batchIndex, $totalBatches) {
                    $this->handleMatchingBatchCompletion($cacheKey, $batchIndex + 1, $totalBatches);
                })
                ->catch(function (\Throwable $e) use ($cacheKey, $batchIndex) {
                    Log::error("Matching batch failed", ['batch' => $batchIndex + 1, 'error' => $e->getMessage()]);
                    $this->handleMatchingBatchCompletion($cacheKey, $batchIndex + 1, $totalBatches);
                })
                ->dispatch();
                
            Log::info("Dispatched matching batch", ['batch' => $batchIndex + 1, 'receipt_ids' => $batch]);
        }
    }
    
    /**
     * Handle matching batch completion and log final completion
     */
    private function handleMatchingBatchCompletion(string $cacheKey, int $completedBatch, int $totalBatches): void
    {
        $data = \Cache::get($cacheKey);
        if (!$data) {
            Log::warning("Matching completion data not found in cache", ['cache_key' => $cacheKey]);
            return;
        }
        
        $data['completed_batches']++;
        \Cache::put($cacheKey, $data, now()->addHours(2));
        
        Log::info("Matching batch completed", [
            'completed_batch' => $completedBatch,
            'total_batches' => $totalBatches,
            'completed_count' => $data['completed_batches']
        ]);
        
        // If all batches are complete, log final completion
        if ($data['completed_batches'] >= $totalBatches) {
            Log::info("All matching batches completed - 2-stage receipt processing finished", [
                'receipt_count' => count($data['receipt_ids'])
            ]);
            
            // Clean up cache
            \Cache::forget($cacheKey);
        }
    }
    
    /**
     * Legacy method for backward compatibility - now uses the new 2-stage workflow
     */
    public function processReceiptsInParallel(array $receiptIds, int $batchSize = 5): void
    {
        $this->processReceiptsOptimized($receiptIds);
    }
    
    /**
     * Legacy method for backward compatibility - now uses the new 2-stage workflow
     */
    public function processMatchingInParallel(array $receiptIds, int $batchSize = 10): void
    {
        $this->processMatchingStage($receiptIds);
    }
}



