<?php

namespace App\Services;

use App\Jobs\ProcessReceipt;
use App\Jobs\ProcessReceiptOCR;
use App\Jobs\ProcessReceiptDeduplication;
use App\Jobs\ProcessReceiptMatching;
use App\Jobs\MatchReceipt;
use App\Models\Receipt;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Log;

class ParallelProcessingService
{
    /**
     * Process multiple receipts in parallel batches
     */
    public function processReceiptsInParallel(array $receiptIds, int $batchSize = 5): void
    {
        $batches = array_chunk($receiptIds, $batchSize);
        
        foreach ($batches as $batch) {
            $jobs = [];
            
            // Create OCR processing jobs
            foreach ($batch as $receiptId) {
                $jobs[] = new ProcessReceipt($receiptId);
            }
            
            // Dispatch batch with delay to prevent overwhelming the API
            Bus::batch($jobs)
                ->name("Process Receipts Batch " . implode(',', $batch))
                ->allowFailures()
                ->onQueue('ocr')
                ->dispatch();
                
            Log::info("Dispatched OCR batch for receipts", ['receipt_ids' => $batch]);
        }
    }
    
    /**
     * Process matching jobs in parallel after OCR is complete
     */
    public function processMatchingInParallel(array $receiptIds, int $batchSize = 10): void
    {
        $batches = array_chunk($receiptIds, $batchSize);
        
        foreach ($batches as $batch) {
            $jobs = [];
            
            // Create matching jobs
            foreach ($batch as $receiptId) {
                $jobs[] = new MatchReceipt($receiptId);
            }
            
            // Dispatch batch
            Bus::batch($jobs)
                ->name("Match Receipts Batch " . implode(',', $batch))
                ->allowFailures()
                ->onQueue('matching')
                ->dispatch();
                
            Log::info("Dispatched matching batch for receipts", ['receipt_ids' => $batch]);
        }
    }
    
    /**
     * Process receipts with optimized parallel workflow
     */
    public function processReceiptsOptimized(array $receiptIds): void
    {
        // First, process OCR in smaller batches for better API rate limiting
        $this->processReceiptsInParallel($receiptIds, 3);
        
        // Schedule matching jobs to run after OCR completes
        // This will be handled by the job completion callbacks
        foreach ($receiptIds as $receiptId) {
            // Schedule matching job with delay to allow OCR to complete
            MatchReceipt::dispatch($receiptId)
                ->delay(now()->addMinutes(2))
                ->onQueue('matching');
        }
    }
}
