<?php

namespace App\Services;

use App\Jobs\ProcessReceipt;
use App\Jobs\ProcessReceiptOCR;
use App\Jobs\ProcessReceiptDeduplication;
use App\Jobs\ProcessReceiptMatching;
use App\Jobs\MatchReceipt;
use App\Models\Receipt;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Log;

class ParallelProcessingService
{
    /**
     * Process multiple receipts with the correct 3-stage workflow:
     * 1. OCR ALL receipts first
     * 2. THEN check for duplicates/grouping
     * 3. THEN match to statements
     */
    public function processReceiptsOptimized(array $receiptIds): void
    {
        Log::info("Starting 3-stage receipt processing", ['receipt_count' => count($receiptIds)]);
        
        // Stage 1: OCR ALL receipts first
        $this->processOCRStage($receiptIds);
    }
    
    /**
     * Stage 1: Process OCR for all receipts in parallel batches
     */
    private function processOCRStage(array $receiptIds): void
    {
        $batches = array_chunk($receiptIds, 3); // Smaller batches for better API rate limiting
        $totalBatches = count($batches);
        
        Log::info("Processing OCR in batches", ['total_batches' => $totalBatches, 'receipt_count' => count($receiptIds)]);
        
        // Create a master batch that contains all OCR batches
        $allOCRJobs = [];
        foreach ($batches as $batchIndex => $batch) {
            $jobs = [];
            
            // Create OCR processing jobs
            foreach ($batch as $receiptId) {
                $jobs[] = new ProcessReceiptOCR($receiptId);
            }
            
            // Create a sub-batch for this group of receipts
            $subBatch = Bus::batch($jobs)
                ->name("OCR Stage - Batch " . ($batchIndex + 1))
                ->allowFailures()
                ->onQueue('ocr');
                
            $allOCRJobs[] = $subBatch;
            
            Log::info("Prepared OCR batch", ['batch' => $batchIndex + 1, 'receipt_ids' => $batch]);
        }
        
        // Dispatch all OCR batches and then trigger deduplication when ALL are complete
        Bus::batch($allOCRJobs)
            ->name("OCR Stage - All Batches")
            ->allowFailures()
            ->onQueue('ocr')
            ->then(function () use ($receiptIds) {
                Log::info("All OCR batches completed, starting deduplication stage", ['receipt_count' => count($receiptIds)]);
                $this->processDeduplicationStage($receiptIds);
            })
            ->catch(function (\Throwable $e) use ($receiptIds) {
                Log::error("OCR stage failed", ['error' => $e->getMessage(), 'receipt_count' => count($receiptIds)]);
            })
            ->dispatch();
            
        Log::info("Dispatched all OCR batches", ['total_batches' => $totalBatches]);
    }
    
    /**
     * Schedule deduplication stage to run after OCR completes
     */
    private function scheduleDeduplicationStage(array $receiptIds): void
    {
        // This will be called by the OCR batch completion callback
        // No immediate action needed here
    }
    
    /**
     * Stage 2: Process deduplication for all receipts
     */
    private function processDeduplicationStage(array $receiptIds): void
    {
        $batches = array_chunk($receiptIds, 5); // Slightly larger batches for deduplication
        
        foreach ($batches as $batchIndex => $batch) {
            $jobs = [];
            
            // Create deduplication jobs
            foreach ($batch as $receiptId) {
                $jobs[] = new ProcessReceiptDeduplication($receiptId);
            }
            
            // Dispatch batch
            Bus::batch($jobs)
                ->name("Deduplication Stage - Batch " . ($batchIndex + 1))
                ->allowFailures()
                ->onQueue('deduplication')
                ->then(function () use ($receiptIds) {
                    Log::info("Deduplication stage completed, scheduling matching", ['receipt_count' => count($receiptIds)]);
                    $this->processMatchingStage($receiptIds);
                })
                ->dispatch();
                
            Log::info("Dispatched deduplication batch", ['batch' => $batchIndex + 1, 'receipt_ids' => $batch]);
        }
    }
    
    /**
     * Schedule matching stage to run after deduplication completes
     */
    private function scheduleMatchingStage(array $receiptIds): void
    {
        // This will be called by the deduplication batch completion callback
        // No immediate action needed here
    }
    
    /**
     * Stage 3: Process matching for all receipts
     */
    private function processMatchingStage(array $receiptIds): void
    {
        $batches = array_chunk($receiptIds, 10); // Larger batches for matching
        
        foreach ($batches as $batchIndex => $batch) {
            $jobs = [];
            
            // Create matching jobs
            foreach ($batch as $receiptId) {
                $jobs[] = new ProcessReceiptMatching($receiptId);
            }
            
            // Dispatch batch
            Bus::batch($jobs)
                ->name("Matching Stage - Batch " . ($batchIndex + 1))
                ->allowFailures()
                ->onQueue('matching')
                ->then(function () use ($receiptIds) {
                    Log::info("Matching stage completed - all receipt processing finished", ['receipt_count' => count($receiptIds)]);
                })
                ->dispatch();
                
            Log::info("Dispatched matching batch", ['batch' => $batchIndex + 1, 'receipt_ids' => $batch]);
        }
    }
    
    /**
     * Legacy method for backward compatibility - now uses the new 3-stage workflow
     */
    public function processReceiptsInParallel(array $receiptIds, int $batchSize = 5): void
    {
        $this->processReceiptsOptimized($receiptIds);
    }
    
    /**
     * Legacy method for backward compatibility - now uses the new 3-stage workflow
     */
    public function processMatchingInParallel(array $receiptIds, int $batchSize = 10): void
    {
        $this->processMatchingStage($receiptIds);
    }
}


