<?php

namespace App\Http\Livewire\Dashboard;

use App\Models\CustomerOrderLineQuantities;
use App\Models\ShipmentLineSizes;
use Livewire\Component;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class GarmentsDueChart extends Component
{
    public $chartData = [];
    public $isLoading = true;

    private const CACHE_DURATION = 24 * 60; // 24 hours in minutes

    protected $listeners = [
        'refreshDashboard' => 'refreshData'
    ];

    public function mount()
    {
        $this->loadChartData();
    }

    public function refreshData()
    {
        $this->isLoading = true;
        // Clear cache to get fresh data
        Cache::forget('garments_due_chart_data');
        $this->loadChartData();
        $this->isLoading = false;
    }

    public function loadChartData()
    {
        $this->chartData = Cache::remember('garments_due_chart_data', self::CACHE_DURATION, function () {
            $currentMonth = Carbon::now()->startOfMonth();
            $startDate = $currentMonth->copy()->subMonths(6);
            $endDate = $currentMonth->copy()->addMonths(6);
            
            $chartData = [
                'labels' => [],
                'datasets' => []
            ];

            // Generate month labels (6 months past to 6 months future)
            for ($i = -6; $i <= 6; $i++) {
                $month = $currentMonth->copy()->addMonths($i);
                $chartData['labels'][] = $month->format('M Y');
            }

            // Get data for 3 years
            $years = [
                ['year' => $currentMonth->year - 2, 'color' => '#dc3545', 'label' => ($currentMonth->year - 2)], // Red for 3rd year
                ['year' => $currentMonth->year - 1, 'color' => '#0d6efd', 'label' => ($currentMonth->year - 1)], // Blue for previous year
                ['year' => $currentMonth->year, 'color' => '#198754', 'label' => $currentMonth->year] // Green for current year
            ];

            foreach ($years as $yearData) {
                $yearData['data'] = [];
                
                // Calculate garments due for each month in this year
                for ($i = -6; $i <= 6; $i++) {
                    $month = $currentMonth->copy()->addMonths($i)->year($yearData['year']);
                    $garmentsDue = $this->getGarmentsDueForMonth($month);
                    $yearData['data'][] = $garmentsDue;
                }
                
                $chartData['datasets'][] = [
                    'label' => $yearData['label'],
                    'data' => $yearData['data'],
                    'backgroundColor' => $yearData['color'],
                    'borderColor' => $yearData['color'],
                    'borderWidth' => 1,
                    'fill' => false,
                    'tension' => 0.1
                ];
            }

            // Debug logging
            \Log::info('Garments Due Chart Data:', $chartData);

            return $chartData;
        });
        
        $this->isLoading = false;
    }

    private function getGarmentsDueForMonth(Carbon $month)
    {
        $startOfMonth = $month->copy()->startOfMonth();
        $endOfMonth = $month->copy()->endOfMonth();

        // Get total garments due for this month by looking at shipment lines with exfty in this month
        // and summing the qty column from shipment_line_sizes
        $garmentsDue = DB::table('shipment_line_sizes')
            ->join('shipment_lines', 'shipment_line_sizes.shipment_line_id', '=', 'shipment_lines.id')
            ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
            ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
            ->whereNull('shipment_line_sizes.deleted_at')
            ->whereNull('shipment_lines.deleted_at')
            ->whereNull('customer_order_lines.deleted_at')
            ->whereNull('customer_orders.deleted_at')
            ->where('customer_orders.cancelled', false)
            ->where('customer_order_lines.cancelled', false)
            ->whereBetween('shipment_lines.exfty', [$startOfMonth, $endOfMonth])
            ->sum('shipment_line_sizes.qty');

        // Debug logging for this specific month
        \Log::info("Garments Due for {$month->format('Y-m')}: Total Qty={$garmentsDue}");

        return $garmentsDue ?? 0;
    }

    public function render()
    {
        return view('livewire.dashboard.garments-due-chart');
    }
}
