const Stripe = require('stripe');
const db = require('./db-multi-tenant');

// Initialize Stripe with secret key
const stripe = process.env.STRIPE_SECRET_KEY 
  ? new Stripe(process.env.STRIPE_SECRET_KEY) 
  : null;

if (!stripe) {
  console.warn('[STRIPE WARNING] STRIPE_SECRET_KEY not set. Stripe functionality will be disabled.');
}

// Cache for subscription tiers from database
let cachedTiers = null;
let cacheTime = null;
const CACHE_DURATION = 5 * 60 * 1000; // 5 minutes

/**
 * Get subscription tiers from database (with caching)
 */
async function getSubscriptionTiers() {
  const now = Date.now();
  
  // Return cached tiers if still valid
  if (cachedTiers && cacheTime && (now - cacheTime < CACHE_DURATION)) {
    return cachedTiers;
  }
  
  try {
    const tiers = await db.getAllSubscriptionTiers();
    
    // Convert database format to service format
    const tiersMap = {};
    for (const tier of tiers) {
      tiersMap[tier.tier_name] = {
        name: tier.display_name,
        maxPhotos: tier.max_photos,
        maxStorageBytes: tier.max_storage_bytes,
        maxAlbums: tier.max_albums,
        customDomain: tier.custom_domain_enabled,
        price: tier.price_monthly,
        priceId: tier.stripe_price_id,
        trialDays: tier.trial_days,
      };
    }
    
    // Update cache
    cachedTiers = tiersMap;
    cacheTime = now;
    
    return tiersMap;
  } catch (error) {
    console.error('[STRIPE] Error fetching tiers from database:', error);
    
    // Fallback to hardcoded values if database fails
    return {
      starter: {
        name: 'Starter',
        maxPhotos: 1000,
        maxStorageBytes: 5 * 1024 * 1024 * 1024,
        maxAlbums: null,
        customDomain: false,
        price: 1000,
        priceId: process.env.STRIPE_PRICE_STARTER || '',
        trialDays: 14,
      },
      professional: {
        name: 'Professional',
        maxPhotos: 10000,
        maxStorageBytes: 50 * 1024 * 1024 * 1024,
        maxAlbums: null,
        customDomain: true,
        price: 2500,
        priceId: process.env.STRIPE_PRICE_PROFESSIONAL || '',
        trialDays: 0,
      },
      business: {
        name: 'Business',
        maxPhotos: 30000,
        maxStorageBytes: 150 * 1024 * 1024 * 1024,
        maxAlbums: null,
        customDomain: true,
        price: 7500,
        priceId: process.env.STRIPE_PRICE_BUSINESS || '',
        trialDays: 0,
      },
    };
  }
}

// Legacy constant for backwards compatibility
const SUBSCRIPTION_TIERS = {};

/**
 * Create a Stripe customer for a new user
 */
async function createCustomer(email, username, fullName) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const customer = await stripe.customers.create({
    email,
    name: fullName || username,
    metadata: {
      username,
    },
  });
  
  return customer.id;
}

/**
 * Create a subscription with free trial
 */
async function createSubscription(customerId, tier, trialDays = 14) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const tierConfig = SUBSCRIPTION_TIERS[tier];
  if (!tierConfig || !tierConfig.priceId) {
    throw new Error(`Invalid tier: ${tier}`);
  }
  
  const subscriptionParams = {
    customer: customerId,
    items: [{ price: tierConfig.priceId }],
    metadata: {
      tier,
    },
    payment_behavior: 'default_incomplete',
    payment_settings: {
      save_default_payment_method: 'on_subscription',
    },
    expand: ['latest_invoice.payment_intent'],
  };
  
  // Add trial period if specified
  if (trialDays > 0) {
    subscriptionParams.trial_period_days = trialDays;
  }
  
  const subscription = await stripe.subscriptions.create(subscriptionParams);
  
  return {
    subscriptionId: subscription.id,
    status: subscription.status,
    clientSecret: subscription.latest_invoice?.payment_intent?.client_secret || null,
    trialEnd: subscription.trial_end ? new Date(subscription.trial_end * 1000) : null,
  };
}

/**
 * Update a subscription to a new tier
 */
async function updateSubscription(subscriptionId, newTier) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const tierConfig = SUBSCRIPTION_TIERS[newTier];
  if (!tierConfig || !tierConfig.priceId) {
    throw new Error(`Invalid tier: ${newTier}`);
  }
  
  // Get current subscription
  const subscription = await stripe.subscriptions.retrieve(subscriptionId);
  
  // Update subscription
  const updated = await stripe.subscriptions.update(subscriptionId, {
    items: [{
      id: subscription.items.data[0].id,
      price: tierConfig.priceId,
    }],
    metadata: {
      tier: newTier,
    },
    proration_behavior: 'create_prorations', // Prorate the cost difference
  });
  
  return {
    subscriptionId: updated.id,
    status: updated.status,
  };
}

/**
 * Cancel a subscription at period end
 */
async function cancelSubscription(subscriptionId) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const subscription = await stripe.subscriptions.update(subscriptionId, {
    cancel_at_period_end: true,
  });
  
  return {
    subscriptionId: subscription.id,
    status: subscription.status,
    cancelAt: subscription.cancel_at ? new Date(subscription.cancel_at * 1000) : null,
  };
}

/**
 * Reactivate a cancelled subscription
 */
async function reactivateSubscription(subscriptionId) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const subscription = await stripe.subscriptions.update(subscriptionId, {
    cancel_at_period_end: false,
  });
  
  return {
    subscriptionId: subscription.id,
    status: subscription.status,
  };
}

/**
 * Create a billing portal session for customer to manage payment methods
 */
async function createBillingPortalSession(customerId, returnUrl) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const session = await stripe.billingPortal.sessions.create({
    customer: customerId,
    return_url: returnUrl,
  });
  
  return session.url;
}

/**
 * Create a checkout session for upgrading (alternative to subscriptions API)
 */
async function createCheckoutSession(customerId, tier, successUrl, cancelUrl, includeTrial = false) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const tierConfig = SUBSCRIPTION_TIERS[tier];
  if (!tierConfig || !tierConfig.priceId) {
    throw new Error(`Invalid tier: ${tier}`);
  }
  
  const sessionParams = {
    customer: customerId,
    mode: 'subscription',
    line_items: [{
      price: tierConfig.priceId,
      quantity: 1,
    }],
    success_url: successUrl,
    cancel_url: cancelUrl,
    subscription_data: {
      metadata: {
        tier,
      },
    },
  };
  
  // Only include trial for Starter plan or if explicitly requested
  if (includeTrial || tier === 'starter') {
    sessionParams.subscription_data.trial_period_days = 14;
  }
  
  const session = await stripe.checkout.sessions.create(sessionParams);
  
  return session.url;
}

/**
 * Get subscription details
 */
async function getSubscription(subscriptionId) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const subscription = await stripe.subscriptions.retrieve(subscriptionId);
  
  return {
    id: subscription.id,
    status: subscription.status,
    tier: subscription.metadata.tier || 'starter',
    currentPeriodEnd: new Date(subscription.current_period_end * 1000),
    cancelAtPeriodEnd: subscription.cancel_at_period_end,
    trialEnd: subscription.trial_end ? new Date(subscription.trial_end * 1000) : null,
  };
}

/**
 * Verify webhook signature
 */
function constructWebhookEvent(payload, signature, webhookSecret) {
  if (!stripe) throw new Error('Stripe not configured');
  
  return stripe.webhooks.constructEvent(payload, signature, webhookSecret);
}

/**
 * Get tier configuration
 */
function getTierConfig(tier) {
  return SUBSCRIPTION_TIERS[tier] || SUBSCRIPTION_TIERS.free;
}

/**
 * Get all tier configurations
 */
function getAllTiers() {
  return SUBSCRIPTION_TIERS;
}

/**
 * Check if Stripe is configured
 */
function isConfigured() {
  return !!stripe && !!STRIPE_PRICES.starter && !!STRIPE_PRICES.professional && !!STRIPE_PRICES.business;
}

module.exports = {
  stripe,
  createCustomer,
  createSubscription,
  updateSubscription,
  cancelSubscription,
  reactivateSubscription,
  createBillingPortalSession,
  createCheckoutSession,
  getSubscription,
  constructWebhookEvent,
  getTierConfig,
  getAllTiers,
  isConfigured,
  SUBSCRIPTION_TIERS,
};

