const Stripe = require('stripe');
const db = require('./db-multi-tenant');

// Initialize Stripe with secret key
const stripe = process.env.STRIPE_SECRET_KEY 
  ? new Stripe(process.env.STRIPE_SECRET_KEY) 
  : null;

if (!stripe) {
  console.warn('[STRIPE WARNING] STRIPE_SECRET_KEY not set. Stripe functionality will be disabled.');
}

// Cache for subscription tiers from database
let cachedTiers = null;
let cacheTime = null;
const CACHE_DURATION = 5 * 60 * 1000; // 5 minutes

/**
 * Get subscription tiers from database (with caching)
 */
async function getSubscriptionTiers() {
  const now = Date.now();
  
  // Return cached tiers if still valid
  if (cachedTiers && cacheTime && (now - cacheTime < CACHE_DURATION)) {
    return cachedTiers;
  }
  
  try {
    const tiers = await db.getAllSubscriptionTiers();
    
    // Convert database format to service format
    const tiersMap = {};
    for (const tier of tiers) {
      tiersMap[tier.tier_key] = {
        name: tier.display_name,
        maxPhotos: tier.max_photos,
        maxStorageBytes: tier.max_storage_bytes,
        maxAlbums: tier.max_albums,
        customDomain: tier.custom_domain_enabled,
        price: tier.price_monthly,
        priceId: tier.stripe_price_id,
        trialDays: tier.trial_days,
      };
    }
    
    // Update cache
    cachedTiers = tiersMap;
    cacheTime = now;
    
    return tiersMap;
  } catch (error) {
    console.error('[STRIPE] Error fetching tiers from database:', error);
    
    // Fallback to hardcoded values if database fails
    return {
      starter: {
        name: 'Starter',
        maxPhotos: 1000,
        maxStorageBytes: 5 * 1024 * 1024 * 1024,
        maxAlbums: null,
        customDomain: false,
        price: 1000,
        priceId: process.env.STRIPE_PRICE_STARTER || '',
        trialDays: 14,
      },
      professional: {
        name: 'Professional',
        maxPhotos: 10000,
        maxStorageBytes: 50 * 1024 * 1024 * 1024,
        maxAlbums: null,
        customDomain: true,
        price: 2500,
        priceId: process.env.STRIPE_PRICE_PROFESSIONAL || '',
        trialDays: 0,
      },
      business: {
        name: 'Business',
        maxPhotos: 30000,
        maxStorageBytes: 150 * 1024 * 1024 * 1024,
        maxAlbums: null,
        customDomain: true,
        price: 7500,
        priceId: process.env.STRIPE_PRICE_BUSINESS || '',
        trialDays: 0,
      },
    };
  }
}

// Legacy constant for backwards compatibility
const SUBSCRIPTION_TIERS = {};

/**
 * Create a Stripe customer for a new user
 */
async function createCustomer(email, username, fullName) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const customer = await stripe.customers.create({
    email,
    name: fullName || username,
    metadata: {
      username,
    },
  });
  
  return customer.id;
}

/**
 * Create a subscription with free trial
 */
async function createSubscription(customerId, tier, trialDays = null) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const tiers = await getSubscriptionTiers();
  const tierConfig = tiers[tier];
  
  if (!tierConfig || !tierConfig.priceId) {
    throw new Error(`Invalid tier: ${tier}`);
  }
  
  // Use tier's default trial days if not specified
  const actualTrialDays = trialDays !== null ? trialDays : (tierConfig.trialDays || 0);
  
  const subscriptionParams = {
    customer: customerId,
    items: [{ price: tierConfig.priceId }],
    metadata: {
      tier,
    },
    payment_behavior: 'default_incomplete',
    payment_settings: {
      save_default_payment_method: 'on_subscription',
    },
    expand: ['latest_invoice.payment_intent'],
  };
  
  // Add trial period if specified
  if (actualTrialDays > 0) {
    subscriptionParams.trial_period_days = actualTrialDays;
  }
  
  const subscription = await stripe.subscriptions.create(subscriptionParams);
  
  return {
    subscriptionId: subscription.id,
    status: subscription.status,
    clientSecret: subscription.latest_invoice?.payment_intent?.client_secret || null,
    trialEnd: subscription.trial_end ? new Date(subscription.trial_end * 1000) : null,
  };
}

/**
 * Update a subscription to a new tier
 */
async function updateSubscription(subscriptionId, newTier) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const tiers = await getSubscriptionTiers();
  const tierConfig = tiers[newTier];
  
  if (!tierConfig || !tierConfig.priceId) {
    throw new Error(`Invalid tier: ${newTier}`);
  }
  
  // Get current subscription
  const subscription = await stripe.subscriptions.retrieve(subscriptionId);
  
  // Update subscription
  const updated = await stripe.subscriptions.update(subscriptionId, {
    items: [{
      id: subscription.items.data[0].id,
      price: tierConfig.priceId,
    }],
    metadata: {
      tier: newTier,
    },
    proration_behavior: 'create_prorations', // Prorate the cost difference
  });
  
  return {
    subscriptionId: updated.id,
    status: updated.status,
  };
}

/**
 * Cancel a subscription at period end
 */
async function cancelSubscription(subscriptionId) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const subscription = await stripe.subscriptions.update(subscriptionId, {
    cancel_at_period_end: true,
  });
  
  return {
    subscriptionId: subscription.id,
    status: subscription.status,
    cancelAt: subscription.cancel_at ? new Date(subscription.cancel_at * 1000) : null,
  };
}

/**
 * Reactivate a cancelled subscription
 */
async function reactivateSubscription(subscriptionId) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const subscription = await stripe.subscriptions.update(subscriptionId, {
    cancel_at_period_end: false,
  });
  
  return {
    subscriptionId: subscription.id,
    status: subscription.status,
  };
}

/**
 * Create a billing portal session for customer to manage payment methods
 */
async function createBillingPortalSession(customerId, returnUrl) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const session = await stripe.billingPortal.sessions.create({
    customer: customerId,
    return_url: returnUrl,
  });
  
  return session.url;
}

/**
 * Create a checkout session for upgrading (alternative to subscriptions API)
 */
async function createCheckoutSession(customerId, tier, successUrl, cancelUrl, includeTrial = false, oldSubscriptionId = null) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const tiers = await getSubscriptionTiers();
  const tierConfig = tiers[tier];
  
  if (!tierConfig || !tierConfig.priceId) {
    throw new Error(`Invalid tier: ${tier}`);
  }
  
  const sessionParams = {
    customer: customerId,
    mode: 'subscription',
    line_items: [{
      price: tierConfig.priceId,
      quantity: 1,
    }],
    success_url: successUrl,
    cancel_url: cancelUrl,
    subscription_data: {
      metadata: {
        tier,
      },
    },
    metadata: {
      tier,
      old_subscription_id: oldSubscriptionId || '', // Store old subscription for cancellation after upgrade
    },
  };
  
  // Use tier's default trial days or explicit override
  const trialDays = includeTrial ? 14 : (tierConfig.trialDays || 0);
  if (trialDays > 0) {
    sessionParams.subscription_data.trial_period_days = trialDays;
  }
  
  const session = await stripe.checkout.sessions.create(sessionParams);
  
  return session.url;
}

/**
 * Get subscription details
 */
async function getSubscription(subscriptionId) {
  if (!stripe) throw new Error('Stripe not configured');
  
  const subscription = await stripe.subscriptions.retrieve(subscriptionId);
  
  return {
    id: subscription.id,
    status: subscription.status,
    tier: subscription.metadata.tier || 'starter',
    currentPeriodEnd: new Date(subscription.current_period_end * 1000),
    cancelAtPeriodEnd: subscription.cancel_at_period_end,
    trialEnd: subscription.trial_end ? new Date(subscription.trial_end * 1000) : null,
  };
}

/**
 * Verify webhook signature
 */
function constructWebhookEvent(payload, signature, webhookSecret) {
  if (!stripe) throw new Error('Stripe not configured');
  
  return stripe.webhooks.constructEvent(payload, signature, webhookSecret);
}

/**
 * Get tier configuration
 */
async function getTierConfig(tier) {
  const tiers = await getSubscriptionTiers();
  return tiers[tier] || null;
}

/**
 * Get all tier configurations
 */
async function getAllTiers() {
  return await getSubscriptionTiers();
}

/**
 * Check if Stripe is configured
 */
async function isConfigured() {
  if (!stripe) return false;
  
  try {
    const tiers = await getSubscriptionTiers();
    // Check that all tiers have stripe_price_id configured
    return !!(tiers.starter?.priceId && tiers.professional?.priceId && tiers.business?.priceId);
  } catch (error) {
    console.error('[STRIPE] Error checking configuration:', error);
    return false;
  }
}

/**
 * Clear the tier cache (useful after updating tiers in database)
 */
function clearTierCache() {
  cachedTiers = null;
  cacheTime = null;
}

/**
 * Get customer details from Stripe with all subscriptions
 */
async function getCustomerWithSubscriptions(customerId) {
  if (!stripe) throw new Error('Stripe not configured');
  
  try {
    const customer = await stripe.customers.retrieve(customerId, {
      expand: ['subscriptions']
    });
    return customer;
  } catch (error) {
    console.error('[STRIPE] Error fetching customer:', error);
    return null;
  }
}

/**
 * List all subscriptions for a customer
 */
async function listCustomerSubscriptions(customerId) {
  if (!stripe) throw new Error('Stripe not configured');
  
  try {
    const subscriptions = await stripe.subscriptions.list({
      customer: customerId,
      limit: 100,
    });
    return subscriptions.data;
  } catch (error) {
    console.error('[STRIPE] Error listing subscriptions:', error);
    return [];
  }
}

module.exports = {
  stripe,
  createCustomer,
  createSubscription,
  updateSubscription,
  cancelSubscription,
  reactivateSubscription,
  createBillingPortalSession,
  createCheckoutSession,
  getSubscription,
  getCustomerWithSubscriptions,
  listCustomerSubscriptions,
  constructWebhookEvent,
  getTierConfig,
  getAllTiers,
  isConfigured,
  getSubscriptionTiers,
  clearTierCache,
  SUBSCRIPTION_TIERS,
};

