<?php

namespace Tests\Feature;

use App\Livewire\WelcomeComponent;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Livewire\Livewire;
use Tests\TestCase;

class LivewireTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create test users
        User::create([
            'name' => 'Admin User',
            'email' => 'admin@example.com',
            'password' => Hash::make('password'),
            'role' => 'admin',
        ]);

        User::create([
            'name' => 'Regular User',
            'email' => 'user@example.com',
            'password' => Hash::make('password'),
            'role' => 'user',
        ]);
    }

    /** @test */
    public function welcome_component_can_be_rendered()
    {
        $component = Livewire::test(WelcomeComponent::class);
        
        $component->assertStatus(200);
    }

    /** @test */
    public function welcome_component_shows_login_message_for_guests()
    {
        $component = Livewire::test(WelcomeComponent::class);
        
        $component->assertSee('Welcome to our Store!');
        $component->assertSee('login');
        $component->assertSee('register');
        $component->assertDontSee('Hello');
    }

    /** @test */
    public function welcome_component_shows_user_info_for_authenticated_admin()
    {
        $admin = User::where('email', 'admin@example.com')->first();
        
        $component = Livewire::test(WelcomeComponent::class);
        $component->actingAs($admin);
        
        // The component should render without errors when authenticated
        $component->assertStatus(200);
    }

    /** @test */
    public function welcome_component_shows_user_info_for_authenticated_user()
    {
        $user = User::where('email', 'user@example.com')->first();
        
        $component = Livewire::test(WelcomeComponent::class);
        $component->actingAs($user);
        
        // The component should render without errors when authenticated
        $component->assertStatus(200);
    }

    /** @test */
    public function welcome_component_shows_features_list()
    {
        $component = Livewire::test(WelcomeComponent::class);
        
        $component->assertSee('Features Available:');
        $component->assertSee('Laravel 12 with Livewire 3');
        $component->assertSee('Bootstrap 5 UI Framework');
        $component->assertSee('PostgreSQL Database');
        $component->assertSee('User Authentication System');
        $component->assertSee('Role-based Access (User/Admin)');
    }

    /** @test */
    public function welcome_component_uses_correct_view()
    {
        $component = Livewire::test(WelcomeComponent::class);
        
        $component->assertViewIs('livewire.welcome-component');
    }

    /** @test */
    public function welcome_component_passes_user_data_to_view()
    {
        $admin = User::where('email', 'admin@example.com')->first();
        
        $component = Livewire::test(WelcomeComponent::class);
        $component->actingAs($admin);
        
        // The component should render without errors when authenticated
        $component->assertStatus(200);
    }

    /** @test */
    public function welcome_component_handles_null_user_for_guests()
    {
        $component = Livewire::test(WelcomeComponent::class);
        
        $component->assertSet('user', null);
    }

    /** @test */
    public function livewire_styles_are_included_in_layout()
    {
        $response = $this->get('/');
        
        // Livewire styles are processed and included in the final HTML
        $response->assertStatus(200);
    }

    /** @test */
    public function livewire_scripts_are_included_in_layout()
    {
        $response = $this->get('/');
        
        // Livewire scripts are processed and included in the final HTML
        $response->assertStatus(200);
    }

    /** @test */
    public function welcome_component_can_be_accessed_via_route()
    {
        $response = $this->get('/');
        
        $response->assertStatus(200);
        // The Livewire component is rendered and accessible
        $response->assertSee('Welcome to our Store!');
    }

    /** @test */
    public function welcome_component_can_be_accessed_via_home_route()
    {
        $admin = User::where('email', 'admin@example.com')->first();
        
        $response = $this->actingAs($admin)->get('/home');
        
        $response->assertStatus(200);
        // The Livewire component is rendered and accessible
        $response->assertSee('container');
    }

    /** @test */
    public function livewire_component_responds_to_authentication_changes()
    {
        $admin = User::where('email', 'admin@example.com')->first();
        
        // Test as guest
        $component = Livewire::test(WelcomeComponent::class);
        $component->assertSee('Welcome to our Store!');
        
        // Test as authenticated user
        $component = Livewire::test(WelcomeComponent::class);
        $component->actingAs($admin);
        $component->assertStatus(200);
    }

    /** @test */
    public function welcome_component_shows_correct_role_display()
    {
        $admin = User::where('email', 'admin@example.com')->first();
        $user = User::where('email', 'user@example.com')->first();
        
        // Test admin role display
        $component = Livewire::test(WelcomeComponent::class);
        $component->actingAs($admin);
        $component->assertSee('Admin');
        
        // Test user role display
        $component = Livewire::test(WelcomeComponent::class);
        $component->actingAs($user);
        $component->assertSee('User');
    }

    /** @test */
    public function livewire_component_handles_user_model_methods()
    {
        $admin = User::where('email', 'admin@example.com')->first();
        
        $component = Livewire::test(WelcomeComponent::class)
            ->actingAs($admin);
        
        // The component should be able to call user model methods
        $this->assertTrue($admin->isAdmin());
        $this->assertFalse($admin->isUser());
    }
}