# NRW Web - Deployment Guide

Complete guide to deploying your NRW Web SaaS platform to Google Cloud Run.

## 🎯 Architecture Overview

```
Cloudflare CDN (*.nrwweb.com)
    ↓
Google Cloud Run (nrw-web service)
    ↓
Google Cloud SQL (PostgreSQL)
    ↓
Google Cloud Storage (photos bucket)
```

## 📋 Prerequisites

1. **Google Cloud account** with billing enabled
2. **Domain name** (e.g., `nrwweb.com`)
3. **Cloudflare account** (free tier)
4. **gcloud CLI** installed

## 🚀 Quick Deploy (5 minutes)

### 1. Set up Google Cloud

```bash
# Set your project
PROJECT_ID="nrw-web-prod"
REGION="europe-west1"
BUCKET_NAME="${PROJECT_ID}-photos"
DB_INSTANCE="photowebsite-db"

gcloud config set project ${PROJECT_ID}

# Enable required APIs
gcloud services enable run.googleapis.com
gcloud services enable storage.googleapis.com
gcloud services enable secretmanager.googleapis.com
gcloud services enable sqladmin.googleapis.com

# Create storage bucket
gcloud storage buckets create gs://${BUCKET_NAME} \
  --location=${REGION} \
  --uniform-bucket-level-access

# Create Cloud SQL PostgreSQL instance
gcloud sql instances create ${DB_INSTANCE} \
  --database-version=POSTGRES_15 \
  --tier=db-f1-micro \
  --region=${REGION} \
  --storage-type=SSD \
  --storage-size=10GB

# Set database root password
DB_PASSWORD=$(openssl rand -base64 32)
gcloud sql users set-password postgres \
  --instance=${DB_INSTANCE} \
  --password="${DB_PASSWORD}"

# Create the database
gcloud sql databases create photowebsite \
  --instance=${DB_INSTANCE}

# Store secrets
echo -n "your-secure-admin-password" | \
  gcloud secrets create ADMIN_PASSWORD \
    --data-file=- \
    --replication-policy=automatic

echo -n "${DB_PASSWORD}" | \
  gcloud secrets create DB_PASSWORD \
    --data-file=- \
    --replication-policy=automatic
```

### 2. Deploy to Cloud Run

```bash
gcloud run deploy nrw-web \
  --source . \
  --region=${REGION} \
  --platform=managed \
  --allow-unauthenticated \
  --memory=512Mi \
  --cpu=1 \
  --max-instances=100 \
  --min-instances=0 \
  --add-cloudsql-instances=${PROJECT_ID}:${REGION}:${DB_INSTANCE} \
  --set-env-vars="USE_POSTGRES=true,USE_CLOUD_STORAGE=true,GCS_BUCKET_NAME=${BUCKET_NAME},DB_NAME=photowebsite,DB_USER=postgres,INSTANCE_UNIX_SOCKET=/cloudsql/${PROJECT_ID}:${REGION}:${DB_INSTANCE},SITE_NAME=NRW Web,PRIMARY_COLOR=#3b82f6,LOGO_URL=/nrw-web.png" \
  --set-secrets="ADMIN_PASSWORD=ADMIN_PASSWORD:latest,DB_PASSWORD=DB_PASSWORD:latest"
```

**Note:** Database tables are created automatically on first run.

### 3. Get your URL

```bash
gcloud run services describe nrw-web \
  --region=${REGION} \
  --format='value(status.url)'
```

You'll get something like: `https://nrw-web-abc123.run.app`

## 🌐 Cloudflare Setup (15 minutes)

### 1. Buy & Add Domain

1. Purchase domain: `nrwweb.com` (£10/year from Namecheap, GoDaddy, etc.)
2. Create Cloudflare account (free)
3. Add site to Cloudflare
4. Update nameservers at your registrar

### 2. Configure DNS

In Cloudflare DNS:

```
Type: CNAME
Name: *
Target: nrw-web-abc123.run.app (your Cloud Run URL without https://)
Proxy: ON (orange cloud ☁️)
```

This creates wildcard subdomain support: `*.nrwweb.com`

### 3. Configure SSL

In Cloudflare SSL/TLS:
- SSL/TLS encryption mode: **Full (strict)**
- Always Use HTTPS: **ON**
- Automatic HTTPS Rewrites: **ON**

## ✅ Verify Setup

Test these URLs (replace with your domain):

1. **Main site:** `https://nrwweb.com` - Should show landing page
2. **Wildcard:** `https://test.nrwweb.com` - Should work
3. **Admin:** `https://nrwweb.com/admin` - Should prompt for password

## 👥 Multi-Tenant Configuration

Your app is now ready for multiple photographers!

### How It Works:

**1. Subdomains (Built-in):**
```
john-smith.nrwweb.com → John's gallery
jane-doe.nrwweb.com   → Jane's gallery
```

Your app checks `req.headers.host` to determine which user.

**2. Custom Domains (via Cloudflare):**

Photographer wants `johnsmithphotography.com`:

```
Customer adds to their DNS:
CNAME www → john-smith.nrwweb.com
```

Done! Their domain works.

### Database Structure:

PostgreSQL tables (created automatically):
- `album_groups` - Groups of albums
- `albums` - Photo albums with settings
- `photos` - Individual photos
- `site_config` - Site-wide configuration

For multi-tenant setup, you'll add a `users` or `tenants` table to track photographers and their subdomains.

## 💳 Adding Stripe (Payments)

### 1. Create Stripe Account

Sign up at stripe.com

### 2. Add Webhook Endpoint

In Stripe Dashboard → Developers → Webhooks:

```
Endpoint URL: https://nrwweb.com/webhook/stripe
Events: customer.subscription.created, customer.subscription.updated
```

### 3. Add Secret to Cloud Run

```bash
echo -n "whsec_your_stripe_webhook_secret" | \
  gcloud secrets create STRIPE_WEBHOOK_SECRET \
    --data-file=- \
    --replication-policy=automatic

gcloud run services update nrw-web \
  --region=${REGION} \
  --set-secrets="STRIPE_WEBHOOK_SECRET=STRIPE_WEBHOOK_SECRET:latest"
```

### 4. Implement Webhook (already in code)

The webhook in `src/server.js` handles:
- Creating user accounts on signup
- Updating tiers on upgrade
- Canceling accounts on churn

## 📊 Monitoring & Costs

### Set Up Budget Alerts

```bash
# Alert at 50%, 75%, 100% of budget
gcloud billing budgets create \
  --billing-account=YOUR-BILLING-ACCOUNT-ID \
  --display-name="NRW Web Monthly Budget" \
  --budget-amount=500GBP \
  --threshold-rule=percent=50 \
  --threshold-rule=percent=75 \
  --threshold-rule=percent=100
```

### Monitor Costs

**Google Cloud Console:** Billing → Cost Breakdown

**Expected costs (100 customers):**
- Cloud Run: £50/month
- Cloud SQL: £15/month (db-f1-micro)
- Cloud Storage: £100/month (50GB avg per customer)
- Network: £200/month (bandwidth)
- **Total: ~£365/month**
- **Revenue: £1,500-2,000/month** (at £15-20 avg per customer)

## 🔒 Security Checklist

- ✅ Secrets in Secret Manager (not environment variables)
- ✅ HTTPS everywhere (Cloudflare + Cloud Run)
- ✅ Rate limiting enabled
- ✅ Admin password is strong
- ✅ CORS properly configured
- ✅ SQL injection prevented (using ORM)
- ✅ XSS prevented (EJS auto-escapes)

## 🔄 Updating the Service

```bash
# Pull latest changes
git pull

# Deploy update
gcloud run deploy nrw-web \
  --source . \
  --region=${REGION}
```

Zero-downtime deployment! Cloud Run handles rolling updates.

## 📈 Scaling

**Auto-scaling is built-in:**
- Scales to 0 when no traffic (cost = £0)
- Scales up automatically under load
- Max 100 instances (configurable)

**To increase limits:**
```bash
gcloud run services update nrw-web \
  --max-instances=500 \
  --cpu=2 \
  --memory=1Gi
```

## 🐛 Troubleshooting

### Issue: "Permission denied"
```bash
# Grant Cloud Run permission to access bucket
gcloud projects add-iam-policy-binding ${PROJECT_ID} \
  --member="serviceAccount:${PROJECT_ID}@appspot.gserviceaccount.com" \
  --role="roles/storage.objectAdmin"
```

### Issue: "Secret not found"
```bash
# List secrets
gcloud secrets list

# Grant access
gcloud secrets add-iam-policy-binding ADMIN_PASSWORD \
  --member="serviceAccount:${PROJECT_ID}@appspot.gserviceaccount.com" \
  --role="roles/secretmanager.secretAccessor"
```

### Issue: "Too many redirects"
In Cloudflare: SSL/TLS → Change to **Full (strict)**

## 🎯 Next Steps

1. ✅ Deploy to Cloud Run
2. ✅ Configure Cloudflare
3. ✅ Test subdomain: `test.nrwweb.com`
4. ✅ Test admin: `nrwweb.com/admin`
5. ⏭️ Set up Stripe
6. ⏭️ Create landing page
7. ⏭️ Get first customer!

## 📚 Additional Resources

- [Google Cloud Run Docs](https://cloud.google.com/run/docs)
- [Cloudflare Docs](https://developers.cloudflare.com/)
- [Stripe Webhooks](https://stripe.com/docs/webhooks)

---

Need help? Check [USAGE-LIMITS-GUIDE.md](./USAGE-LIMITS-GUIDE.md) for tier configuration and cost control.
