<?php

namespace App\Jobs;

use App\Models\StatementTransaction;
use App\Models\Receipt;
use App\Services\AI\ExpenseClassificationAgent;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ReclassifyWithReceipt implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public StatementTransaction $transaction,
        public Receipt $receipt
    ) {}

    public function handle(ExpenseClassificationAgent $agent): void
    {
        try {
            $result = $agent->reclassifyWithReceipt($this->transaction, $this->receipt);
            
            Log::info('Transaction reclassified with receipt', [
                'transaction_id' => $this->transaction->id,
                'receipt_id' => $this->receipt->id,
                'merchant' => $this->transaction->merchant_name,
                'result' => $result,
            ]);

            // Check if splitting is suggested
            if (!empty($result['split_suggestion'])) {
                $this->handleTransactionSplitting($agent);
            }

        } catch (\Exception $e) {
            Log::error('Failed to reclassify transaction with receipt', [
                'transaction_id' => $this->transaction->id,
                'receipt_id' => $this->receipt->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            
            $this->fail($e);
        }
    }

    /**
     * Handle transaction splitting based on receipt analysis
     */
    private function handleTransactionSplitting(ExpenseClassificationAgent $agent): void
    {
        $splitAnalysis = $agent->shouldSplitTransaction($this->transaction, $this->receipt);
        
        if ($splitAnalysis['should_split'] && !empty($splitAnalysis['split_lines'])) {
            Log::info('Transaction splitting suggested during reclassification', [
                'transaction_id' => $this->transaction->id,
                'receipt_id' => $this->receipt->id,
                'reasoning' => $splitAnalysis['reasoning'],
                'split_lines' => $splitAnalysis['split_lines'],
            ]);

            // Store split suggestion in transaction meta
            $this->transaction->update([
                'meta' => array_merge($this->transaction->meta ?? [], [
                    'split_suggestion' => $splitAnalysis,
                    'split_analyzed_at' => now()->toISOString(),
                    'split_triggered_by_receipt' => $this->receipt->id,
                ])
            ]);
        }
    }

    /**
     * Handle job failure
     */
    public function failed(\Throwable $exception): void
    {
        Log::error('ReclassifyWithReceipt job failed', [
            'transaction_id' => $this->transaction->id,
            'receipt_id' => $this->receipt->id,
            'error' => $exception->getMessage(),
        ]);
    }
}
