<?php

namespace App\Livewire\Manage;

use App\Models\Receipt;
use App\Models\ReceiptGroup;
use App\Services\AI\ReceiptGroupingService;
use Livewire\Component;
use Livewire\WithPagination;

class ReceiptGrouping extends Component
{
    use WithPagination;

    public $selectedReceipts = [];
    public $groupName = '';
    public $groupDescription = '';
    public $showCreateGroupModal = false;
    public $showSuggestionsModal = false;
    public $aiSuggestions = [];
    public $selectedGroup = null;
    public $showGroupDetails = false;

    protected $rules = [
        'groupName' => 'required|string|max:255',
        'groupDescription' => 'nullable|string|max:1000',
    ];

    public function mount()
    {
        $this->loadAiSuggestions();
    }

    public function render()
    {
        $unmatchedReceipts = Receipt::whereNull('receipt_group_id')
            ->whereDoesntHave('matches')
            ->with(['lines', 'department', 'account'])
            ->orderBy('receipt_date', 'desc')
            ->paginate(20);

        $receiptGroups = ReceiptGroup::with(['receipts.lines', 'receipts.matches'])
            ->orderBy('created_at', 'desc')
            ->paginate(10, ['*'], 'groups_page');

        return view('livewire.manage.receipt-grouping', [
            'unmatchedReceipts' => $unmatchedReceipts,
            'receiptGroups' => $receiptGroups,
        ]);
    }

    public function selectReceipt($receiptId)
    {
        if (in_array($receiptId, $this->selectedReceipts)) {
            $this->selectedReceipts = array_diff($this->selectedReceipts, [$receiptId]);
        } else {
            $this->selectedReceipts[] = $receiptId;
        }
    }

    public function selectAllReceipts()
    {
        $receiptIds = Receipt::whereNull('receipt_group_id')
            ->whereDoesntHave('matches')
            ->pluck('id')
            ->toArray();
        
        $this->selectedReceipts = $receiptIds;
    }

    public function clearSelection()
    {
        $this->selectedReceipts = [];
    }

    public function showCreateGroupModal()
    {
        if (empty($this->selectedReceipts)) {
            session()->flash('error', 'Please select at least one receipt to group.');
            return;
        }

        $this->groupName = '';
        $this->groupDescription = '';
        $this->showCreateGroupModal = true;
    }

    public function createGroup()
    {
        $this->validate();

        $group = ReceiptGroup::create([
            'name' => $this->groupName,
            'description' => $this->groupDescription,
            'grouping_method' => 'manual',
            'user_id' => auth()->id(),
        ]);

        Receipt::whereIn('id', $this->selectedReceipts)
            ->update(['receipt_group_id' => $group->id]);

        $this->showCreateGroupModal = false;
        $this->selectedReceipts = [];
        $this->groupName = '';
        $this->groupDescription = '';

        session()->flash('success', 'Receipt group created successfully.');
    }

    public function loadAiSuggestions()
    {
        $unmatchedReceipts = Receipt::whereNull('receipt_group_id')
            ->whereDoesntHave('matches')
            ->with(['lines'])
            ->get();

        if ($unmatchedReceipts->count() < 2) {
            $this->aiSuggestions = [];
            return;
        }

        $groupingService = app(ReceiptGroupingService::class);
        $this->aiSuggestions = $groupingService->suggestGroupings($unmatchedReceipts);
    }

    public function showSuggestionsModal()
    {
        $this->loadAiSuggestions();
        $this->showSuggestionsModal = true;
    }

    public function applySuggestion($suggestionIndex)
    {
        if (!isset($this->aiSuggestions[$suggestionIndex])) {
            return;
        }

        $suggestion = $this->aiSuggestions[$suggestionIndex];
        $groupingService = app(ReceiptGroupingService::class);
        
        $group = $groupingService->createGroupFromSuggestion($suggestion, auth()->id());
        
        // Remove this suggestion from the list
        unset($this->aiSuggestions[$suggestionIndex]);
        $this->aiSuggestions = array_values($this->aiSuggestions);

        session()->flash('success', 'AI suggestion applied successfully.');
    }

    public function viewGroup($groupId)
    {
        $this->selectedGroup = ReceiptGroup::with(['receipts.lines', 'receipts.matches.statementTransaction'])
            ->find($groupId);
        $this->showGroupDetails = true;
    }

    public function ungroupReceipt($receiptId)
    {
        Receipt::where('id', $receiptId)->update(['receipt_group_id' => null]);
        
        // If this was the last receipt in the group, delete the group
        $group = $this->selectedGroup;
        if ($group && $group->receipts()->count() <= 1) {
            $group->delete();
            $this->showGroupDetails = false;
            $this->selectedGroup = null;
            session()->flash('success', 'Receipt ungrouped and group deleted.');
        } else {
            $this->selectedGroup->refresh();
            session()->flash('success', 'Receipt ungrouped successfully.');
        }
    }

    public function deleteGroup($groupId)
    {
        $group = ReceiptGroup::find($groupId);
        if ($group) {
            // Ungroup all receipts first
            $group->receipts()->update(['receipt_group_id' => null]);
            $group->delete();
            session()->flash('success', 'Group deleted successfully.');
        }
    }

    public function addReceiptToGroup($receiptId, $groupId)
    {
        Receipt::where('id', $receiptId)->update(['receipt_group_id' => $groupId]);
        $this->selectedGroup->refresh();
        session()->flash('success', 'Receipt added to group successfully.');
    }

    public function removeReceiptFromGroup($receiptId)
    {
        Receipt::where('id', $receiptId)->update(['receipt_group_id' => null]);
        $this->selectedGroup->refresh();
        session()->flash('success', 'Receipt removed from group successfully.');
    }

    public function closeModals()
    {
        $this->showCreateGroupModal = false;
        $this->showSuggestionsModal = false;
        $this->showGroupDetails = false;
        $this->selectedGroup = null;
    }
}