<?php

namespace App\Jobs;

use App\Models\Receipt;
use App\Services\ReceiptMergingService;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ProcessMergedReceipt implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $timeout = 120;
    public int $maxExceptions = 2;

    public function __construct(public int $mergedReceiptId) {}

    public function backoff(): array
    {
        return [15, 60, 300];
    }

    public function retryUntil(): \DateTimeInterface
    {
        return now()->addHours(6);
    }

    public function handle(ReceiptMergingService $mergingService): void
    {
        try {
            $mergedReceipt = Receipt::find($this->mergedReceiptId);
            if (!$mergedReceipt) {
                \Log::warning('Merged receipt not found', ['receipt_id' => $this->mergedReceiptId]);
                return;
            }

            // Re-run OCR and classification on the merged receipt
            $mergingService->reprocessMergedReceipt($mergedReceipt);

            \Log::info('Merged receipt processed successfully', [
                'merged_receipt_id' => $this->mergedReceiptId,
                'merchant_name' => $mergedReceipt->merchant_name,
                'total_amount' => $mergedReceipt->total_amount
            ]);

        } catch (\Throwable $e) {
            \Log::warning('ProcessMergedReceipt: transient failure, will retry', [
                'merged_receipt_id' => $this->mergedReceiptId,
                'error' => $e->getMessage(),
                'attempt' => method_exists($this, 'attempts') ? $this->attempts() : null,
            ]);
            $schedule = (array) $this->backoff();
            $index = max(0, min((int)($this->attempts() - 1), count($schedule) - 1));
            $this->release((int) $schedule[$index]);
        }
    }
}

