<?php

namespace App\Http\Controllers;

use App\Models\CustomRequest;
use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use App\Models\SafetyEvent;
use App\Services\CustomSessionService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AdminController extends Controller
{
    public function __construct(
        private CustomSessionService $customSessionService
    ) {
        $this->middleware('auth');
        $this->middleware(function ($request, $next) {
            if (!Auth::user()->isAdmin()) {
                abort(403, 'Unauthorized');
            }
            return $next($request);
        });
    }

    public function index()
    {
        $stats = [
            'total_categories' => LibraryCategory::count(),
            'total_tracks' => LibraryTrack::count(),
            'pending_requests' => CustomRequest::byStatus('pending')->count(),
            'open_safety_events' => SafetyEvent::open()->count(),
        ];

        $recentRequests = CustomRequest::byStatus('pending')
            ->with('user')
            ->latest()
            ->limit(5)
            ->get();

        $recentSafetyEvents = SafetyEvent::open()
            ->with('user')
            ->latest()
            ->limit(5)
            ->get();

        return view('admin.index', compact('stats', 'recentRequests', 'recentSafetyEvents'));
    }

    public function categories()
    {
        $categories = LibraryCategory::withCount('tracks')->orderBy('sort_order')->get();
        return view('admin.categories.index', compact('categories'));
    }

    public function tracks()
    {
        $tracks = LibraryTrack::with('category')->latest()->paginate(20);
        return view('admin.tracks.index', compact('tracks'));
    }

    public function customRequests()
    {
        $requests = CustomRequest::with('user')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.custom-requests.index', compact('requests'));
    }

    public function approveRequest(int $id)
    {
        $request = CustomRequest::findOrFail($id);
        
        if ($request->status !== 'pending') {
            return back()->with('error', 'Request cannot be approved.');
        }

        $this->customSessionService->approveRequest($request);

        return back()->with('success', 'Request approved successfully.');
    }

    public function rejectRequest(Request $request, int $id)
    {
        $request->validate([
            'reason' => 'required|string|max:500',
        ]);

        $customRequest = CustomRequest::findOrFail($id);
        
        if ($customRequest->status !== 'pending') {
            return back()->with('error', 'Request cannot be rejected.');
        }

        $this->customSessionService->rejectRequest($customRequest, $request->reason);

        return back()->with('success', 'Request rejected successfully.');
    }

    public function safetyEvents()
    {
        $events = SafetyEvent::with(['user', 'resolvedBy'])
            ->latest()
            ->paginate(20);

        return view('admin.safety-events.index', compact('events'));
    }

    public function resolveSafetyEvent(Request $request, int $id)
    {
        $request->validate([
            'resolution_notes' => 'required|string|max:1000',
        ]);

        $event = SafetyEvent::findOrFail($id);
        $user = Auth::user();

        $event->markAsResolved($request->resolution_notes, $user);

        return back()->with('success', 'Safety event resolved successfully.');
    }

    public function closeSafetyEvent(int $id)
    {
        $event = SafetyEvent::findOrFail($id);
        $event->markAsClosed();

        return back()->with('success', 'Safety event closed successfully.');
    }
}
