<?php

namespace App\Http\Controllers;

use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use App\Models\SafetyEvent;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Str;

class AdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('admin');
    }

    public function index(): View
    {
        $stats = [
            'categories' => LibraryCategory::count(),
            'tracks' => LibraryTrack::count(),
            'activeTracks' => LibraryTrack::where('is_active', true)->count(),
            'safetyEvents' => SafetyEvent::where('status', '!=', 'resolved')->count(),
        ];

        $recentEvents = SafetyEvent::with('user')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        return view('admin.index', compact('stats', 'recentEvents'));
    }

    public function categories(): View
    {
        $categories = LibraryCategory::withCount('tracks')
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        return view('admin.categories', compact('categories'));
    }

    public function tracks(): View
    {
        $tracks = LibraryTrack::with('category')
            ->orderBy('title')
            ->paginate(20);

        $categories = LibraryCategory::orderBy('name')->get();

        return view('admin.tracks', compact('tracks', 'categories'));
    }

    public function safetyEvents(): View
    {
        $events = SafetyEvent::with(['user', 'resolvedBy'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.safety-events', compact('events'));
    }

    public function storeCategory(Request $request): RedirectResponse
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:library_categories,name',
            'description' => 'nullable|string|max:1000',
            'icon' => 'nullable|string|max:100',
            'color' => 'nullable|string|max:7|regex:/^#[0-9A-F]{6}$/i',
            'sort_order' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);

        LibraryCategory::create([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'description' => $request->description,
            'icon' => $request->icon,
            'color' => $request->color ?? '#4dbda0',
            'sort_order' => $request->sort_order ?? 0,
            'is_active' => $request->is_active ?? true,
        ]);

        return redirect()->route('admin.categories')
            ->with('success', 'Category created successfully!');
    }

    public function updateCategory(Request $request, LibraryCategory $category): RedirectResponse
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:library_categories,name,' . $category->id,
            'description' => 'nullable|string|max:1000',
            'icon' => 'nullable|string|max:100',
            'color' => 'nullable|string|max:7|regex:/^#[0-9A-F]{6}$/i',
            'sort_order' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);

        $category->update([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'description' => $request->description,
            'icon' => $request->icon,
            'color' => $request->color,
            'sort_order' => $request->sort_order,
            'is_active' => $request->is_active,
        ]);

        return redirect()->route('admin.categories')
            ->with('success', 'Category updated successfully!');
    }

    public function destroyCategory(LibraryCategory $category): RedirectResponse
    {
        if ($category->tracks()->count() > 0) {
            return redirect()->route('admin.categories')
                ->with('error', 'Cannot delete category with existing tracks!');
        }

        $category->delete();

        return redirect()->route('admin.categories')
            ->with('success', 'Category deleted successfully!');
    }

    public function storeTrack(Request $request): RedirectResponse
    {
        $request->validate([
            'category_id' => 'required|exists:library_categories,id',
            'title' => 'required|string|max:255|unique:library_tracks,title',
            'description' => 'required|string|max:1000',
            'transcript' => 'required|string',
            'duration' => 'required|integer|min:60|max:7200', // 1 min to 2 hours
            'file_path' => 'required|string|max:500',
            'file_size' => 'required|integer|min:1',
            'voice_type' => 'required|string|in:male,female,neutral',
            'difficulty_level' => 'required|string|in:beginner,intermediate,advanced',
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:50',
            'safety_notes' => 'nullable|array',
            'safety_notes.*' => 'string|max:200',
            'is_active' => 'boolean',
        ]);

        LibraryTrack::create([
            'category_id' => $request->category_id,
            'title' => $request->title,
            'slug' => Str::slug($request->title),
            'description' => $request->description,
            'transcript' => $request->transcript,
            'duration' => $request->duration,
            'file_path' => $request->file_path,
            'file_size' => $request->file_size,
            'voice_type' => $request->voice_type,
            'difficulty_level' => $request->difficulty_level,
            'tags' => $request->tags ?? [],
            'safety_notes' => $request->safety_notes ?? [],
            'is_active' => $request->is_active ?? true,
            'play_count' => 0,
            'rating' => 0.0,
        ]);

        return redirect()->route('admin.tracks')
            ->with('success', 'Track created successfully!');
    }

    public function updateTrack(Request $request, LibraryTrack $track): RedirectResponse
    {
        $request->validate([
            'category_id' => 'required|exists:library_categories,id',
            'title' => 'required|string|max:255|unique:library_tracks,title,' . $track->id,
            'description' => 'required|string|max:1000',
            'transcript' => 'required|string',
            'duration' => 'required|integer|min:60|max:7200',
            'file_path' => 'required|string|max:500',
            'file_size' => 'required|integer|min:1',
            'voice_type' => 'required|string|in:male,female,neutral',
            'difficulty_level' => 'required|string|in:beginner,intermediate,advanced',
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:50',
            'safety_notes' => 'nullable|array',
            'safety_notes.*' => 'string|max:200',
            'is_active' => 'boolean',
        ]);

        $track->update([
            'category_id' => $request->category_id,
            'title' => $request->title,
            'slug' => Str::slug($request->title),
            'description' => $request->description,
            'transcript' => $request->transcript,
            'duration' => $request->duration,
            'file_path' => $request->file_path,
            'file_size' => $request->file_size,
            'voice_type' => $request->voice_type,
            'difficulty_level' => $request->difficulty_level,
            'tags' => $request->tags ?? [],
            'safety_notes' => $request->safety_notes ?? [],
            'is_active' => $request->is_active,
        ]);

        return redirect()->route('admin.tracks')
            ->with('success', 'Track updated successfully!');
    }

    public function destroyTrack(LibraryTrack $track): RedirectResponse
    {
        $track->delete();

        return redirect()->route('admin.tracks')
            ->with('success', 'Track deleted successfully!');
    }
}
