<?php

namespace App\Http\Controllers;

use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use App\Models\CustomRequest;
use App\Models\SafetyEvent;
use App\Services\CustomSessionService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AdminController extends Controller
{
    public function __construct(
        private CustomSessionService $customSessionService
    ) {
        $this->middleware('auth');
        $this->middleware(function ($request, $next) {
            if (!Auth::user()->isAdmin()) {
                abort(403, 'Admin access required.');
            }
            return $next($request);
        });
    }

    public function index()
    {
        $stats = [
            'categories' => LibraryCategory::count(),
            'tracks' => LibraryTrack::count(),
            'pendingRequests' => CustomRequest::where('status', 'pending')->count(),
            'safetyEvents' => SafetyEvent::where('status', 'open')->count(),
        ];
        
        $recentRequests = CustomRequest::with('user')
            ->where('status', 'pending')
            ->latest()
            ->limit(5)
            ->get();
            
        $recentSafetyEvents = SafetyEvent::with('user')
            ->where('status', 'open')
            ->latest()
            ->limit(5)
            ->get();
        
        return view('admin.index', compact('stats', 'recentRequests', 'recentSafetyEvents'));
    }

    public function categories()
    {
        $categories = LibraryCategory::withCount('tracks')->orderBy('sort_order')->get();
        
        return view('admin.categories', compact('categories'));
    }

    public function tracks()
    {
        $tracks = LibraryTrack::with('category')->orderBy('created_at', 'desc')->paginate(20);
        
        return view('admin.tracks', compact('tracks'));
    }

    public function customRequests()
    {
        $requests = $this->customSessionService->getPendingRequests();
        
        return view('admin.custom-requests', compact('requests'));
    }

    public function approveRequest(int $id)
    {
        $request = CustomRequest::findOrFail($id);
        
        if ($request->status !== 'pending') {
            return back()->with('error', 'Only pending requests can be approved.');
        }
        
        $this->customSessionService->approveRequest($request);
        
        return back()->with('success', 'Request approved successfully.');
    }

    public function rejectRequest(Request $request, int $id)
    {
        $customRequest = CustomRequest::findOrFail($id);
        
        $request->validate([
            'reason' => 'required|string|max:500',
        ]);
        
        if ($customRequest->status !== 'pending') {
            return back()->with('error', 'Only pending requests can be rejected.');
        }
        
        $this->customSessionService->rejectRequest($customRequest, $request->reason);
        
        return back()->with('success', 'Request rejected successfully.');
    }

    public function safetyEvents()
    {
        $events = SafetyEvent::with('user')
            ->orderBy('created_at', 'desc')
            ->paginate(20);
        
        return view('admin.safety-events', compact('events'));
    }

    public function resolveSafetyEvent(Request $request, int $id)
    {
        $event = SafetyEvent::findOrFail($id);
        
        $request->validate([
            'notes' => 'sometimes|nullable|string|max:1000',
        ]);
        
        if ($event->isResolved()) {
            return back()->with('error', 'This safety event has already been resolved.');
        }
        
        $event->markAsResolved($request->notes);
        
        return back()->with('success', 'Safety event resolved successfully.');
    }
}
