<?php

namespace App\Http\Controllers;

use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use App\Models\CustomRequest;
use App\Models\SafetyEvent;
use App\Services\CustomSessionService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AdminController extends Controller
{
    public function __construct(
        private CustomSessionService $customSessionService
    ) {
        $this->middleware('auth');
        $this->middleware(function ($request, $next) {
            if (!Auth::user()->isAdmin()) {
                abort(403, 'Admin access required.');
            }
            return $next($request);
        });
    }

    public function index()
    {
        $stats = [
            'categories' => LibraryCategory::count(),
            'tracks' => LibraryTrack::count(),
            'pendingRequests' => CustomRequest::where('status', 'pending')->count(),
            'safetyEvents' => SafetyEvent::where('status', 'open')->count(),
        ];
        
        $recentRequests = CustomRequest::with('user')
            ->where('status', 'pending')
            ->latest()
            ->limit(5)
            ->get();
            
        $recentSafetyEvents = SafetyEvent::with('user')
            ->where('status', 'open')
            ->latest()
            ->limit(5)
            ->get();
        
        return view('admin.index', compact('stats', 'recentRequests', 'recentSafetyEvents'));
    }

    public function categories()
    {
        $categories = LibraryCategory::withCount('tracks')->orderBy('sort_order')->get();
        
        return view('admin.categories', compact('categories'));
    }

    public function tracks()
    {
        $tracks = LibraryTrack::with('category')->orderBy('created_at', 'desc')->paginate(20);
        $categories = LibraryCategory::orderBy('name')->get();
        
        return view('admin.tracks', compact('tracks', 'categories'));
    }

    public function storeTrack(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string|max:1000',
            'category_id' => 'required|exists:library_categories,id',
            'voice_type' => 'required|in:female,male,neutral',
            'difficulty_level' => 'required|in:beginner,intermediate,advanced',
            'audio_file' => 'required|file|mimes:mp3,wav,m4a|max:51200', // 50MB max
            'transcript' => 'nullable|string|max:5000',
            'tags' => 'nullable|string|max:500',
            'is_active' => 'boolean',
        ]);

        try {
            $mediaService = app(\App\Services\MediaService::class);
            
            // Store the audio file
            $filePath = $mediaService->storeFile($request->file('audio_file'), 'library-tracks');
            
            // Process tags
            $tags = $request->filled('tags') ? array_map('trim', explode(',', $request->tags)) : [];
            
            // Create the track
            $track = LibraryTrack::create([
                'title' => $request->title,
                'slug' => \Str::slug($request->title),
                'description' => $request->description,
                'category_id' => $request->category_id,
                'voice_type' => $request->voice_type,
                'difficulty_level' => $request->difficulty_level,
                'file_path' => $filePath,
                'transcript' => $request->transcript,
                'tags' => $tags,
                'is_active' => $request->boolean('is_active', true),
                'duration' => $mediaService->getAudioDuration($filePath),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Track created successfully',
                'track' => $track
            ]);

        } catch (\Exception $e) {
            \Log::error('Failed to create track', [
                'error' => $e->getMessage(),
                'request_data' => $request->except(['audio_file'])
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to create track: ' . $e->getMessage()
            ], 500);
        }
    }

    public function updateTrack(Request $request, int $id)
    {
        $track = LibraryTrack::findOrFail($id);
        
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string|max:1000',
            'category_id' => 'required|exists:library_categories,id',
            'voice_type' => 'required|in:female,male,neutral',
            'difficulty_level' => 'required|in:beginner,intermediate,advanced',
            'audio_file' => 'nullable|file|mimes:mp3,wav,m4a|max:51200',
            'transcript' => 'nullable|string|max:5000',
            'tags' => 'nullable|string|max:500',
            'is_active' => 'boolean',
        ]);

        try {
            $mediaService = app(\App\Services\MediaService::class);
            
            $updateData = [
                'title' => $request->title,
                'slug' => \Str::slug($request->title),
                'description' => $request->description,
                'category_id' => $request->category_id,
                'voice_type' => $request->voice_type,
                'difficulty_level' => $request->difficulty_level,
                'transcript' => $request->transcript,
                'tags' => $request->filled('tags') ? array_map('trim', explode(',', $request->tags)) : [],
                'is_active' => $request->boolean('is_active', true),
            ];

            // Handle new audio file if provided
            if ($request->hasFile('audio_file')) {
                // Delete old file
                if ($track->file_path) {
                    $mediaService->deleteFile($track->file_path);
                }
                
                // Store new file
                $filePath = $mediaService->storeFile($request->file('audio_file'), 'library-tracks');
                $updateData['file_path'] = $filePath;
                $updateData['duration'] = $mediaService->getAudioDuration($filePath);
            }

            $track->update($updateData);

            return response()->json([
                'success' => true,
                'message' => 'Track updated successfully',
                'track' => $track
            ]);

        } catch (\Exception $e) {
            \Log::error('Failed to update track', [
                'track_id' => $id,
                'error' => $e->getMessage(),
                'request_data' => $request->except(['audio_file'])
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to update track: ' . $e->getMessage()
            ], 500);
        }
    }

    public function destroyTrack(int $id)
    {
        $track = LibraryTrack::findOrFail($id);
        
        try {
            // Delete the audio file
            if ($track->file_path) {
                $mediaService = app(\App\Services\MediaService::class);
                $mediaService->deleteFile($track->file_path);
            }
            
            $track->delete();

            return response()->json([
                'success' => true,
                'message' => 'Track deleted successfully'
            ]);

        } catch (\Exception $e) {
            \Log::error('Failed to delete track', [
                'track_id' => $id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete track: ' . $e->getMessage()
            ], 500);
        }
    }

    public function customRequests()
    {
        $requests = $this->customSessionService->getPendingRequests();
        
        return view('admin.custom-requests', compact('requests'));
    }

    public function approveRequest(int $id)
    {
        $request = CustomRequest::findOrFail($id);
        
        if ($request->status !== 'pending') {
            return back()->with('error', 'Only pending requests can be approved.');
        }
        
        $this->customSessionService->approveRequest($request);
        
        return back()->with('success', 'Request approved successfully.');
    }

    public function rejectRequest(Request $request, int $id)
    {
        $customRequest = CustomRequest::findOrFail($id);
        
        $request->validate([
            'reason' => 'required|string|max:500',
        ]);
        
        if ($customRequest->status !== 'pending') {
            return back()->with('error', 'Only pending requests can be rejected.');
        }
        
        $this->customSessionService->rejectRequest($customRequest, $request->reason);
        
        return back()->with('success', 'Request rejected successfully.');
    }

    public function safetyEvents()
    {
        $events = SafetyEvent::with('user')
            ->orderBy('created_at', 'desc')
            ->paginate(20);
        
        return view('admin.safety-events', compact('events'));
    }

    public function resolveSafetyEvent(Request $request, int $id)
    {
        $event = SafetyEvent::findOrFail($id);
        
        $request->validate([
            'notes' => 'sometimes|nullable|string|max:1000',
        ]);
        
        if ($event->isResolved()) {
            return back()->with('error', 'This safety event has already been resolved.');
        }
        
        $event->markAsResolved($request->notes);
        
        return back()->with('success', 'Safety event resolved successfully.');
    }
}
