<?php

namespace App\Livewire\Manage;

use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Validation\Rules\File;
use Illuminate\Support\Facades\Storage;
use App\Models\Receipt;
use App\Models\ReceiptLine;
use App\Services\ReceiptProcessingService;
use App\Services\ImageResizeService;
use App\Services\FileProcessingService;
use App\Services\ParallelProcessingService;
use App\Jobs\ProcessReceipt;
use App\Jobs\ProcessEMLReceipt;

class ReceiptBulkUpload extends Component
{
    use WithFileUploads;

    public array $files = [];

    protected function rules(): array
    {
        return [
            'files.*' => [File::types(['jpg', 'jpeg', 'png', 'gif', 'eml', 'pdf'])->max('10mb')],
        ];
    }

    public function updatedFiles()
    {
        $this->validateOnly('files.*');
    }

    public function saveUploads(ReceiptProcessingService $processingService, ImageResizeService $imageResize, FileProcessingService $fileProcessor, ParallelProcessingService $parallelProcessor)
    {
        try {
            $this->validate();
            
            if (empty($this->files)) {
                session()->flash('error', 'Please select at least one file to upload.');
                return;
            }

            $disk = config('filesystems.default');
            $uploadedCount = 0;
            $receiptIds = [];
            $emlPaths = [];
            
            foreach ($this->files as $upload) {
                $extension = strtolower($upload->getClientOriginalExtension());
                $path = null;
                $hash = null;
                
                if (in_array($extension, ['jpg', 'jpeg', 'png', 'gif'])) {
                    // Handle image files
                    $filename = uniqid() . '.' . $upload->getClientOriginalExtension();
                    $path = 'receipts/' . $filename;
                    
                    // Resize and store the image
                    $imageResize->resizeAndStore($upload, $path, $disk);
                    $hash = hash_file('sha256', Storage::disk($disk)->path($path));
                } elseif (in_array($extension, ['eml', 'pdf'])) {
                    // Handle EML and PDF files - keep original extension
                    $filename = uniqid() . '.' . $upload->getClientOriginalExtension();
                    
                    if ($extension === 'eml') {
                        $path = 'emails/' . $filename;
                        $upload->storeAs('emails', $filename, $disk);
                        $emlPaths[] = Storage::disk($disk)->path($path);
                    } else { // pdf
                        $path = 'receipts/' . $filename;
                        $upload->storeAs('receipts', $filename, $disk);
                        $hash = hash_file('sha256', Storage::disk($disk)->path($path));
                        
                        $receipt = Receipt::create([
                            'user_id' => auth()->id(),
                            'receipt_date' => now()->toDateString(), // Temporary date, will be updated by OCR
                            'merchant_name' => null, // Will be extracted by OCR
                            'total_amount' => 0, // Will be extracted by OCR
                            'currency' => 'GBP', // Will be extracted by OCR
                            'image_disk' => $disk,
                            'image_path' => $path,
                            'image_hash' => $hash,
                        ]);

                        $receiptIds[] = $receipt->id;
                    }
                    
                    $uploadedCount++;
                }

                // Create receipt for image files
                if (in_array($extension, ['jpg', 'jpeg', 'png', 'gif'])) {
                    $receipt = Receipt::create([
                        'user_id' => auth()->id(),
                        'receipt_date' => now()->toDateString(), // Temporary date, will be updated by OCR
                        'merchant_name' => null, // Will be extracted by OCR
                        'total_amount' => 0, // Will be extracted by OCR
                        'currency' => 'GBP', // Will be extracted by OCR
                        'image_disk' => $disk,
                        'image_path' => $path,
                        'image_hash' => $hash,
                    ]);

                    $receiptIds[] = $receipt->id;
                }
            }

            // Process all receipts in parallel for better performance
            if (!empty($receiptIds)) {
                $parallelProcessor->processReceiptsOptimized($receiptIds);
            }
            
            // Process EML files separately
            foreach ($emlPaths as $emlPath) {
                ProcessEMLReceipt::dispatch($emlPath);
            }

            $this->reset('files');
            session()->flash('status', "Successfully uploaded {$uploadedCount} receipt" . ($uploadedCount > 1 ? 's' : '') . ". Processing will continue in background with optimized parallel processing.");
            
        } catch (\Exception $e) {
            session()->flash('error', 'Upload failed: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.manage.receipt-bulk-upload');
    }
}


