<?php

namespace App\Livewire\Manage;

use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Validation\Rules\File;
use Illuminate\Support\Facades\Storage;
use App\Models\Receipt;
use App\Models\ReceiptLine;
use App\Services\ReceiptProcessingService;
use App\Services\ImageResizeService;
use App\Services\FileProcessingService;
use App\Services\ParallelProcessingService;
use App\Jobs\ProcessReceipt;
use App\Jobs\ProcessEMLReceipt;

class ReceiptBulkUpload extends Component
{
    use WithFileUploads;

    public array $files = [];

    protected function rules(): array
    {
        return [
            'files.*' => [File::types(['jpg', 'jpeg', 'png', 'gif', 'eml', 'pdf'])->max('10mb')],
        ];
    }

    public function updatedFiles()
    {
        $this->validateOnly('files.*');
    }

    public function saveUploads(ReceiptProcessingService $processingService, ImageResizeService $imageResize, FileProcessingService $fileProcessor, ParallelProcessingService $parallelProcessor)
    {
        try {
            $this->validate();
            
            if (empty($this->files)) {
                session()->flash('error', 'Please select at least one file to upload.');
                return;
            }

            $disk = config('filesystems.default');
            $uploadedCount = 0;
            $receiptIds = [];
            $emlPaths = [];
            
            foreach ($this->files as $upload) {
                $extension = strtolower($upload->getClientOriginalExtension());
                $filename = uniqid() . '.' . $upload->getClientOriginalExtension();
                $receiptData = [
                    'user_id' => auth()->id(),
                    'receipt_date' => null,
                    'merchant_name' => null,
                    'total_amount' => 0,
                    'currency' => 'GBP',
                    'image_disk' => $disk,
                ];

                if (in_array($extension, ['jpg', 'jpeg', 'png', 'gif'])) {
                    $path = 'receipts/' . $filename;
                    $imageResize->resizeAndStore($upload, $path, $disk);
                    
                    $receiptData['image_path'] = $path;
                    $receiptData['image_hash'] = hash_file('sha256', Storage::disk($disk)->path($path));
                    
                    $receipt = Receipt::create($receiptData);
                    $receiptIds[] = $receipt->id;
                    $uploadedCount++;

                } elseif ($extension === 'pdf') {
                    $path = 'receipts/' . $filename;
                    $upload->storeAs('receipts', $filename, $disk);

                    $receiptData['image_path'] = $path;
                    $receiptData['image_hash'] = hash_file('sha256', Storage::disk($disk)->path($path));

                    $receipt = Receipt::create($receiptData);
                    $receiptIds[] = $receipt->id;
                    $uploadedCount++;

                } elseif ($extension === 'eml') {
                    $path = 'emails/' . $filename;
                    $upload->storeAs('emails', $filename, $disk);
                    $emlPaths[] = Storage::disk($disk)->path($path);
                    $uploadedCount++;
                }
            }

            // Process all receipts in parallel for better performance
            if (!empty($receiptIds)) {
                $parallelProcessor->processReceiptsOptimized($receiptIds);
            }
            
            // Process EML files separately
            foreach ($emlPaths as $emlPath) {
                ProcessEMLReceipt::dispatch($emlPath, auth()->id());
            }

            $this->reset('files');
            session()->flash('status', "Successfully uploaded {$uploadedCount} " . \Str::plural('file', $uploadedCount) . ". Processing will continue in the background.");
            
        } catch (\Exception $e) {
            session()->flash('error', 'Upload failed: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.manage.receipt-bulk-upload');
    }
}


