<?php

namespace App\Models;

use App\Casts\TitleCase;
use App\Casts\Upper;
use App\Models\BaseModel;
use App\Models\CustomerInvoiceLayout;
use Bkwld\Cloner\Cloneable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Cache;
use OwenIt\Auditing\Contracts\Auditable;

/**
 * Customer model representing business customers.
 * 
 * @package App\Models
 */
class Customer extends BaseModel implements Auditable
{
    use HasFactory;
    use SoftDeletes;
    use \OwenIt\Auditing\Auditable;
    use Cloneable;

    public static function boot()
    {
        parent::boot();

        static::updated(function ($customer) {
            if ($customer->isDirty(['currency'])) {
                Cache::tags(["prices"])->flush();
            }
            self::invalidatePriceResolutions($customer);
        });

        static::saved(function ($customer) {
            self::invalidatePriceResolutions($customer);
        });

        static::deleted(function ($customer) {
            self::invalidatePriceResolutions($customer);
        });

        static::deleting(function($model) {
            $model->styles?->each?->delete();
            $model->customer_orders?->each?->delete();
            $model->customer_agents?->each?->delete();
        });
        static::restoring(function($model) {
            $model->styles()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
            $model->customer_orders()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
            $model->customer_agents()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
        });
        static::addGlobalScope('order', function (Builder $builder) {
            $builder->orderBy('name');
        });
    }

    protected static function booted()
    {
        static::updated(function ($customer) {
            ShipmentLine::where('customer_id', $customer->id)->update([
                'customer_id' => $customer->id,
            ]);
        });
    }

    protected $casts = [
        'name' => TitleCase::class,
        'contact' => TitleCase::class,
        'finance_contact' => TitleCase::class,
        'vat' => Upper::class,
        'eori' => Upper::class,
        'settings' => 'array',
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];



    protected $fillable = [
        'name',
        'countries_id',
        'contact',
        'email',
        'phone',
        'finance_contact',
        'finance_email',
        'finance_phone',
        'notes',
        'finance_notes',
        'payment_terms',
        'factored',
        'currency',
        'created_at',
        'updated_at',
        'settings',
        'custom_prompt',
        'show_mid',
        'logo',
        'show_embroidery',
        'customs',
        'yc_email',
        'yc_address',
        'vat',
        'eori',
        'samples_required',
        'deleted_at',
        'zoho_customer',
        'zoho_customer_tag',
        'zoho_factored',
        'default_commission',
        'default_discount',
    ];



    public function countries()
    {
        return $this->belongsTo(Countries::class);
    }

    public function customer_invoice_layouts()
    {
        return $this->hasMany(CustomerInvoiceLayout::class, 'customer_id');
    }

    public function styles()
    {
        return $this->hasMany(Styles::class, 'customers_id');
    }
    public function designs()
    {
        return $this->hasMany(Design::class, 'customers_id');
    }
    public function customer_addresses()
    {
        return $this->hasMany(CustomerAddress::class);
    }
    public function customer_agents()
    {
        return $this->hasMany(CustomerAgents::class, 'customers_id');
    }
    public function customer_orders()
    {
        return $this->hasMany(CustomerOrders::class, 'customers_id');
    }
    public function shipment_lines()
    {
        return $this->hasMany(ShipmentLines::class);
    }

    public function customer_payment_terms()
    {
        return $this->hasMany(CustomerPaymentTerm::class);
    }

        /**
     * Retrieve a setting with a given name or fall back to the default.
     *
     */
    public function setting(string $name, $default = null)
    {
        if ($this->settings != NULL && array_key_exists($name, $this->settings)) {
            return $this->settings[$name];
        }
        return [];
    }

    /**
     * Update one or more settings and then optionally save the model.
     *
     */
    public function settings(array $revisions, bool $save = true) : self
    {
        if($this->settings != NULL)
            $this->settings = array_merge($this->settings, $revisions);
        else
            $this->settings = $revisions;
        if ($save) {
            $this->save();
        }
        return $this;
    }

    public function scopeSearch($query, $value){
		$query->where('name', 'like', "%{$value}%")
        ->orWhere('id', $value)
		->orWhere('currency', $value)
		->orWhereRelation('countries', 'country', 'like', "%{$value}%")
		->orWhereRelation('countries', 'code', 'like', "%{$value}%");
	}

    protected static function invalidatePriceResolutions($customer)
    {
        try {
            $service = app(\App\Services\PriceResolutionService::class);
            $service->invalidateByCustomer($customer->id);
        } catch (\Exception $e) {
            \Log::error('Failed to invalidate price resolutions for customer', [
                'customer_id' => $customer->id,
                'error' => $e->getMessage()
            ]);
        }
    }
}
