<?php

namespace App\Models;

use App\Casts\Upper;
use App\Models\Design;
use App\Casts\TitleCase;
use App\Models\BaseModel;
use Bkwld\Cloner\Cloneable;
use Illuminate\Support\Str;
use App\Models\CustomerInvoiceLayout;
use Illuminate\Support\Facades\Cache;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Customer extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use Cloneable;

    /**
     * The "booted" method of the model.
     * 
     * @return void
     */
    protected static function booted(): void
    {
        static::updated(function ($customer) {
            if ($customer->isDirty(['currency'])) {
                Cache::tags(["prices"])->flush();
            }
            self::invalidatePriceResolutions($customer);
        });

        static::saved(function ($customer) {
            self::invalidatePriceResolutions($customer);
        });

        static::deleted(function ($customer) {
            self::invalidatePriceResolutions($customer);
        });

        static::deleting(function ($model) {
            $model->styles?->each?->delete();
            $model->customer_orders?->each?->delete();
            $model->customer_agents?->each?->delete();
        });

        static::restoring(function ($model) {
            $model->styles()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
            $model->customer_orders()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
            $model->customer_agents()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
        });

        static::addGlobalScope('order', function (Builder $builder) {
            $builder->orderBy('name');
        });

        static::updated(function ($customer) {
            ShipmentLine::where('customer_id', $customer->id)->update([
                'customer_id' => $customer->id,
            ]);
        });
    }

    protected $casts = [
        'name' => TitleCase::class,
        'contact' => TitleCase::class,
        'finance_contact' => TitleCase::class,
        'vat' => Upper::class,
        'eori' => Upper::class,
        'settings' => 'array',
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];



    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'countries_id',
        'contact',
        'email',
        'phone',
        'finance_contact',
        'finance_email',
        'finance_phone',
        'notes',
        'finance_notes',
        'payment_terms',
        'factored',
        'currency',
        'settings',
        'custom_prompt',
        'show_mid',
        'logo',
        'show_embroidery',
        'customs',
        'yc_email',
        'yc_address',
        'vat',
        'eori',
        'samples_required',
        'zoho_customer',
        'zoho_customer_tag',
        'zoho_factored',
        'default_commission',
        'default_discount',
    ];



    /**
     * Get the country for this customer.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function countries()
    {
        return $this->belongsTo(Countries::class);
    }

    /**
     * Get the customer invoice layouts for this customer.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function customer_invoice_layouts()
    {
        return $this->hasMany(CustomerInvoiceLayout::class, 'customer_id');
    }

    /**
     * Get the styles for this customer.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function styles()
    {
        return $this->hasMany(Styles::class, 'customers_id');
    }

    /**
     * Get the designs for this customer.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function designs()
    {
        return $this->hasMany(Design::class, 'customers_id');
    }

    /**
     * Get the customer addresses for this customer.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function customer_addresses()
    {
        return $this->hasMany(CustomerAddress::class);
    }

    /**
     * Get the customer agents for this customer.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function customer_agents()
    {
        return $this->hasMany(CustomerAgents::class, 'customers_id');
    }

    /**
     * Get the customer orders for this customer.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function customer_orders()
    {
        return $this->hasMany(CustomerOrders::class, 'customers_id');
    }

    /**
     * Get the shipment lines for this customer.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function shipment_lines()
    {
        return $this->hasMany(ShipmentLines::class);
    }

    /**
     * Get the customer payment terms for this customer.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function customer_payment_terms()
    {
        return $this->hasMany(CustomerPaymentTerm::class);
    }

    /**
     * Retrieve a setting with a given name or fall back to the default.
     * 
     * @param string $name
     * @param mixed $default
     * @return mixed
     */
    public function setting(string $name, $default = null)
    {
        if ($this->settings != null && array_key_exists($name, $this->settings)) {
            return $this->settings[$name];
        }
        return $default;
    }

    /**
     * Update one or more settings and then optionally save the model.
     * 
     * @param array $revisions
     * @param bool $save
     * @return self
     */
    public function settings(array $revisions, bool $save = true): self
    {
        if ($this->settings != null) {
            $this->settings = array_merge($this->settings, $revisions);
        } else {
            $this->settings = $revisions;
        }
        
        if ($save) {
            $this->save();
        }
        
        return $this;
    }

    /**
     * Scope a query to search for customers.
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $value
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSearch($query, string $value)
    {
        return $query->where('name', 'like', "%{$value}%")
            ->orWhere('id', $value)
            ->orWhere('currency', $value)
            ->orWhereRelation('countries', 'country', 'like', "%{$value}%")
            ->orWhereRelation('countries', 'code', 'like', "%{$value}%");
    }

    /**
     * Invalidate price resolutions for a customer.
     * 
     * @param \App\Models\Customer $customer
     * @return void
     */
    protected static function invalidatePriceResolutions($customer): void
    {
        try {
            $service = app(\App\Services\PriceResolutionService::class);
            $service->invalidateByCustomer($customer->id);
        } catch (\Exception $e) {
            \Log::error('Failed to invalidate price resolutions for customer', [
                'customer_id' => $customer->id,
                'error' => $e->getMessage()
            ]);
        }
    }
}
