<?php

namespace App\Livewire\Admin;

use App\Models\Item;
use App\Models\Season;
use App\Models\Category;
use Livewire\Component;
use Livewire\WithPagination;

class Items extends Component
{
    use WithPagination;

    public $search = '';
    public $seasonFilter = '';
    public $categoryFilter = '';
    public $statusFilter = '';
    public $perPage = 12;

    protected $queryString = [
        'search' => ['except' => ''],
        'seasonFilter' => ['except' => ''],
        'categoryFilter' => ['except' => ''],
        'statusFilter' => ['except' => ''],
        'perPage' => ['except' => 12],
    ];

    public function mount()
    {
        // Set default season to the most recent active season
        if (empty($this->seasonFilter)) {
            $latestSeason = Season::where('is_active', true)
                ->orderBy('sort_order')
                ->orderBy('created_at', 'desc')
                ->first();
            
            if ($latestSeason) {
                $this->seasonFilter = $latestSeason->id;
            }
        }
        
        // Set default category to null (all categories)
        if (empty($this->categoryFilter)) {
            $this->categoryFilter = '';
        }
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingSeasonFilter()
    {
        $this->resetPage();
    }

    public function updatingCategoryFilter()
    {
        $this->resetPage();
    }

    public function updatingStatusFilter()
    {
        $this->resetPage();
    }

    public function deleteItem($itemId)
    {
        $item = Item::findOrFail($itemId);
        
        $itemName = $item->name;
        
        // Delete associated colourways and their images
        foreach ($item->colourways as $colourway) {
            if ($colourway->image) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($colourway->image);
            }
            $colourway->delete();
        }
        
        // Delete the item
        $item->delete();
        
        session()->flash('message', "Item '{$itemName}' and its colourways have been deleted successfully!");
    }

    public function render()
    {
        $query = Item::with(['season', 'category']);

        if ($this->search) {
            $query->where(function ($q) {
                $q->where('name', 'like', '%' . $this->search . '%')
                  ->orWhere('description', 'like', '%' . $this->search . '%')
                  ->orWhere('sku', 'like', '%' . $this->search . '%');
            });
        }

        if ($this->seasonFilter) {
            $query->where('seasons_id', $this->seasonFilter);
        }

        if ($this->categoryFilter) {
            $query->where('categories_id', $this->categoryFilter);
        }

        if ($this->statusFilter) {
            if ($this->statusFilter === 'active') {
                $query->where('is_active', true);
            } elseif ($this->statusFilter === 'inactive') {
                $query->where('is_active', false);
            }
        }

        $items = $query->orderBy('created_at', 'desc')
                      ->paginate($this->perPage);

        $seasons = Season::where('is_active', true)
                        ->orderBy('sort_order')
                        ->orderBy('name')
                        ->get();

        $categories = Category::where('is_active', true)
                             ->orderBy('sort_order')
                             ->orderBy('name')
                             ->get();

        return view('livewire.admin.items', [
            'items' => $items,
            'seasons' => $seasons,
            'categories' => $categories,
        ])
            ->layout('layouts.admin')
            ->title('Items');
    }
}
