#!/usr/bin/env node

/**
 * Fix User Limits Script
 * Updates all users' max_photos and max_storage_bytes based on their current subscription_tier
 */

require('dotenv').config();
const { Pool } = require('pg');

const dbConfig = process.env.INSTANCE_UNIX_SOCKET
  ? {
      user: process.env.DB_USER || 'postgres',
      password: process.env.DB_PASSWORD,
      database: process.env.DB_NAME || 'photowebsite',
      host: process.env.INSTANCE_UNIX_SOCKET,
    }
  : {
      user: process.env.DB_USER || 'postgres',
      password: process.env.DB_PASSWORD,
      database: process.env.DB_NAME || 'photowebsite',
      host: process.env.DB_HOST || 'localhost',
      port: parseInt(process.env.DB_PORT || '5432', 10),
    };

const pool = new Pool(dbConfig);

const TIER_LIMITS = {
  starter: {
    maxPhotos: 1000,
    maxStorageBytes: 5 * 1024 * 1024 * 1024, // 5GB
  },
  professional: {
    maxPhotos: 10000,
    maxStorageBytes: 50 * 1024 * 1024 * 1024, // 50GB
  },
  business: {
    maxPhotos: 30000,
    maxStorageBytes: 150 * 1024 * 1024 * 1024, // 150GB
  }
};

async function fixUserLimits() {
  try {
    console.log('🔧 Fixing user limits based on subscription tiers...\n');

    // Get all users
    const result = await pool.query('SELECT id, username, subscription_tier, max_photos, max_storage_bytes FROM users');
    const users = result.rows;

    console.log(`Found ${users.length} user(s)\n`);

    for (const user of users) {
      const limits = TIER_LIMITS[user.subscription_tier];
      
      if (!limits) {
        console.log(`⚠️  User ${user.username} has invalid tier: ${user.subscription_tier}`);
        continue;
      }

      const needsUpdate = 
        user.max_photos !== limits.maxPhotos || 
        user.max_storage_bytes !== limits.maxStorageBytes;

      if (needsUpdate) {
        console.log(`📝 Updating ${user.username}:`);
        console.log(`   Tier: ${user.subscription_tier}`);
        console.log(`   Photos: ${user.max_photos} → ${limits.maxPhotos}`);
        console.log(`   Storage: ${(user.max_storage_bytes / 1024 / 1024 / 1024).toFixed(1)}GB → ${(limits.maxStorageBytes / 1024 / 1024 / 1024).toFixed(1)}GB`);

        await pool.query(
          'UPDATE users SET max_photos = $1, max_storage_bytes = $2, updated_at = CURRENT_TIMESTAMP WHERE id = $3',
          [limits.maxPhotos, limits.maxStorageBytes, user.id]
        );

        console.log(`   ✅ Updated!\n`);
      } else {
        console.log(`✓ ${user.username} limits are correct (${user.subscription_tier})\n`);
      }
    }

    console.log('✅ All user limits fixed!');
    
  } catch (error) {
    console.error('❌ Error fixing user limits:', error);
    process.exit(1);
  } finally {
    await pool.end();
  }
}

fixUserLimits();

