<?php

namespace App\Services;

use App\Models\CustomRequest;
use App\Models\CustomSession;
use App\Models\User;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Pagination\LengthAwarePaginator;

class CustomSessionService
{
    public function createRequest(array $data, User $user): CustomRequest
    {
        return CustomRequest::create([
            'user_id' => $user->id,
            'topic' => $data['topic'],
            'description' => $data['description'],
            'session_length' => $data['session_length'] ?? '15min',
            'voice_preference' => $data['voice_preference'] ?? 'female',
            'special_notes' => $data['special_notes'] ?? null,
        ]);
    }

    public function getUserRequests(User $user, array $filters = []): LengthAwarePaginator
    {
        $query = CustomRequest::byUser($user->id)->with('customSession');

        if (isset($filters['status'])) {
            $query->byStatus($filters['status']);
        }

        return $query->orderBy('created_at', 'desc')->paginate(10);
    }

    public function getUserSessions(User $user, array $filters = []): LengthAwarePaginator
    {
        $query = CustomSession::byUser($user->id)->with('customRequest');

        if (isset($filters['status'])) {
            $query->byStatus($filters['status']);
        }

        return $query->orderBy('created_at', 'desc')->paginate(10);
    }

    public function getSession(int $sessionId, User $user): ?CustomSession
    {
        return CustomSession::where('id', $sessionId)
            ->byUser($user->id)
            ->with('customRequest')
            ->first();
    }

    public function processCustomRequest(CustomRequest $request): void
    {
        // In a real app, this would queue a job for AI generation
        // For demo purposes, we'll create a fake session immediately
        
        $request->update(['status' => 'approved']);
        
        CustomSession::create([
            'custom_request_id' => $request->id,
            'user_id' => $request->user_id,
            'title' => "Custom Session: {$request->topic}",
            'description' => $request->description,
            'audio_path' => '/demo/audio/custom-session-' . $request->id . '.mp3',
            'transcript' => $this->generateFakeTranscript($request),
            'duration' => $request->session_length,
            'status' => 'ready',
            'generated_at' => now(),
            'generation_metadata' => [
                'ai_model' => 'demo-model-v1',
                'generation_time' => '2.5s',
                'voice_used' => $request->voice_preference,
            ],
        ]);
    }

    public function getSessionAudioUrl(CustomSession $session): string
    {
        // In a real app, this would generate a signed URL for GCS/S3
        // For demo purposes, we'll return the path directly
        return $session->audio_path;
    }

    private function generateFakeTranscript(CustomRequest $request): string
    {
        $topics = [
            'stress' => 'Welcome to this guided meditation for stress relief. Take a deep breath and let your mind settle...',
            'sleep' => 'This session will help you relax and prepare for a peaceful night\'s sleep. Find a comfortable position...',
            'anxiety' => 'Let\'s work together to calm your anxious thoughts. Remember, it\'s okay to feel this way...',
            'focus' => 'Clear your mind and sharpen your focus. This session will help you achieve mental clarity...',
            'gratitude' => 'Take a moment to reflect on the things you\'re grateful for. Gratitude opens your heart...',
        ];

        $topic = strtolower($request->topic);
        foreach ($topics as $key => $transcript) {
            if (str_contains($topic, $key)) {
                return $transcript;
            }
        }

        return 'Welcome to your personalized wellness session. Take a moment to settle into a comfortable position and let your mind become still...';
    }

    public function getPendingRequests(): Collection
    {
        return CustomRequest::byStatus('pending')->with('user')->get();
    }

    public function approveRequest(CustomRequest $request): void
    {
        $request->update(['status' => 'approved']);
        $this->processCustomRequest($request);
    }

    public function rejectRequest(CustomRequest $request, string $reason): void
    {
        $request->update([
            'status' => 'rejected',
            'admin_notes' => $reason,
        ]);
    }
}
