<?php

namespace App\Services;

use App\Models\CustomRequest;
use App\Models\CustomSession;
use App\Models\User;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class CustomSessionService
{
    public function createCustomRequest(array $data, User $user): CustomRequest
    {
        return CustomRequest::create([
            'user_id' => $user->id,
            'topic' => $data['topic'],
            'description' => $data['description'],
            'session_length' => $data['session_length'] ?? '15min',
            'voice_preference' => $data['voice_preference'] ?? 'neutral',
            'special_notes' => $data['special_notes'] ?? null,
        ]);
    }

    public function generateCustomSession(CustomRequest $request): CustomSession
    {
        // Create the custom session record
        $session = CustomSession::create([
            'user_id' => $request->user_id,
            'custom_request_id' => $request->id,
            'title' => "Custom Session: {$request->topic}",
            'description' => $request->description,
            'voice_type' => $request->voice_preference,
            'status' => 'queued',
        ]);

        // Queue the generation job
        dispatch(function () use ($session) {
            $this->processSessionGeneration($session);
        })->delay(now()->addSeconds(5)); // Simulate processing time

        return $session;
    }

    public function processSessionGeneration(CustomSession $session): void
    {
        try {
            // Update status to generating
            $session->update(['status' => 'generating']);

            // Simulate AI generation process
            $this->simulateAIGeneration($session);

            // Mark as ready
            $session->update([
                'status' => 'ready',
                'generated_at' => now(),
            ]);

        } catch (\Exception $e) {
            $session->update([
                'status' => 'failed',
                'failed_at' => now(),
                'failure_reason' => $e->getMessage(),
            ]);
        }
    }

    private function simulateAIGeneration(CustomSession $session): void
    {
        // Simulate AI processing time
        sleep(rand(2, 5));

        // Generate fake transcript
        $transcript = $this->generateFakeTranscript($session);
        
        // Generate fake audio file path
        $filePath = "custom_sessions/{$session->id}/session_{$session->id}.mp3";
        
        // Update session with generated content
        $session->update([
            'transcript' => $transcript,
            'file_path' => $filePath,
            'file_size' => rand(5000000, 15000000), // 5-15 MB
            'duration' => rand(900, 3600), // 15-60 minutes
            'generation_metadata' => [
                'ai_model' => 'hypno-ai-v1',
                'generation_time' => now()->toISOString(),
                'parameters' => [
                    'voice_type' => $session->voice_type,
                    'session_length' => $session->customRequest->session_length ?? '15min',
                ],
            ],
        ]);
    }

    private function generateFakeTranscript(CustomSession $session): string
    {
        $topic = $session->customRequest->topic ?? 'relaxation';
        
        $transcripts = [
            'relaxation' => "Welcome to your personalized relaxation session. Take a deep breath and let yourself sink into a state of deep calm. Feel the tension melting away from your body as you focus on your breathing. With each breath, you're becoming more relaxed, more peaceful, more centered. Allow yourself to drift into a state of complete tranquility...",
            'sleep' => "As you prepare for sleep, let your mind and body naturally unwind. Feel the weight of the day lifting from your shoulders. Your breathing is becoming slower and deeper, your muscles are relaxing completely. You're entering a state of deep, restorative sleep. Each breath brings you closer to peaceful slumber...",
            'stress' => "Let's release the stress and tension that's been building up. Feel the pressure melting away as you focus on your breathing. With each exhale, you're letting go of worry and anxiety. Your mind is becoming clearer, your body more relaxed. You're creating space for peace and calm to flow through you...",
            'confidence' => "You are strong, capable, and confident. Feel this inner strength growing within you. Every challenge you've faced has made you stronger. You have everything you need to succeed. Your confidence is building with each breath. You're stepping into your power, ready to achieve your goals...",
        ];

        return $transcripts[$topic] ?? $transcripts['relaxation'];
    }

    public function getUserSessions(User $user): \Illuminate\Database\Eloquent\Collection
    {
        return CustomSession::where('user_id', $user->id)
            ->with('customRequest')
            ->orderBy('created_at', 'desc')
            ->get();
    }

    public function getSessionById(int $id, User $user): ?CustomSession
    {
        return CustomSession::where('id', $id)
            ->where('user_id', $user->id)
            ->with('customRequest')
            ->first();
    }

    public function incrementPlayCount(CustomSession $session): void
    {
        $session->increment('play_count');
    }

    public function getSignedUrl(CustomSession $session): ?string
    {
        if (!$session->file_path || $session->status !== 'ready') {
            return null;
        }

        // In a real app, this would generate a signed URL for cloud storage
        // For demo purposes, we'll return a placeholder
        return route('custom.sessions.show', $session->id) . '?audio=true';
    }
}
