<?php

namespace App\Services;

use App\Models\CustomRequest;
use App\Models\CustomSession;
use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class CustomSessionService
{
    public function createRequest(array $data, User $user): CustomRequest
    {
        return CustomRequest::create([
            'user_id' => $user->id,
            'topic' => $data['topic'],
            'description' => $data['description'],
            'duration_minutes' => $data['duration_minutes'],
            'voice_preference' => $data['voice_preference'] ?? 'female',
            'special_notes' => $data['special_notes'] ?? null,
            'goals' => $data['goals'] ?? [],
        ]);
    }

    public function approveRequest(CustomRequest $request): void
    {
        $request->update(['status' => 'approved']);
        
        // Create a custom session and queue it for generation
        $this->createCustomSession($request);
    }

    public function rejectRequest(CustomRequest $request, string $reason): void
    {
        $request->update([
            'status' => 'rejected',
            'admin_notes' => $reason,
        ]);
    }

    public function createCustomSession(CustomRequest $request): CustomSession
    {
        $title = "Custom Session: " . $request->topic;
        
        return CustomSession::create([
            'custom_request_id' => $request->id,
            'user_id' => $request->user_id,
            'title' => $title,
            'description' => $request->description,
            'status' => 'queued',
        ]);
    }

    public function generateSession(CustomSession $session): void
    {
        $session->markAsGenerating();

        try {
            // Simulate AI generation process
            $this->simulateAIGeneration($session);
            
            $session->markAsReady();
        } catch (\Exception $e) {
            $session->markAsFailed();
            throw $e;
        }
    }

    protected function simulateAIGeneration(CustomSession $session): void
    {
        // Simulate processing time
        sleep(2);

        // Generate fake transcript
        $transcript = $this->generateFakeTranscript($session->customRequest);
        
        // Generate fake audio path (in real app, this would be the actual generated audio)
        $audioPath = 'custom-sessions/' . Str::uuid() . '.mp3';
        
        // Generate fake image path
        $imagePath = 'custom-sessions/' . Str::uuid() . '.jpg';
        
        // Calculate duration based on request
        $durationSeconds = $session->customRequest->duration_minutes * 60;
        
        $session->update([
            'transcript' => $transcript,
            'audio_path' => $audioPath,
            'image_path' => $imagePath,
            'duration_seconds' => $durationSeconds,
            'generation_metadata' => [
                'generated_at' => now()->toISOString(),
                'model_version' => 'demo-1.0',
                'processing_time' => 2.0,
            ],
        ]);
    }

    protected function generateFakeTranscript(CustomRequest $request): string
    {
        $topics = [
            'stress' => 'Welcome to this guided meditation for stress relief. Take a deep breath and let yourself relax...',
            'sleep' => 'As you prepare for sleep, let your mind drift gently. Feel the weight of the day lifting...',
            'anxiety' => 'In this session, we\'ll work together to calm your anxious thoughts. Remember, you are safe...',
            'focus' => 'Clear your mind and focus on your breath. With each inhale, feel your concentration deepening...',
        ];

        $topic = strtolower($request->topic);
        $baseText = $topics[$topic] ?? 'Welcome to this personalized wellness session. Let\'s begin our journey together...';
        
        // Add some variety based on duration
        $duration = $request->duration_minutes;
        if ($duration > 10) {
            $baseText .= ' We\'ll take our time with this practice, allowing each moment to unfold naturally...';
        }
        
        if ($request->voice_preference === 'male') {
            $baseText .= ' Let the gentle guidance of this voice lead you deeper into relaxation...';
        }
        
        return $baseText;
    }

    public function getUserSessions(User $user, ?string $status = null): \Illuminate\Database\Eloquent\Collection
    {
        $query = CustomSession::where('user_id', $user->id)->with('customRequest');
        
        if ($status) {
            $query->byStatus($status);
        }
        
        return $query->orderBy('created_at', 'desc')->get();
    }

    public function getSessionById(int $id, User $user): ?CustomSession
    {
        return CustomSession::where('id', $id)
            ->where('user_id', $user->id)
            ->with('customRequest')
            ->first();
    }

    public function getPendingRequests(): \Illuminate\Database\Eloquent\Collection
    {
        return CustomRequest::byStatus('pending')
            ->with('user')
            ->orderBy('created_at', 'asc')
            ->get();
    }

    public function getSignedAudioUrl(CustomSession $session): string
    {
        // In a real app, this would generate a signed URL for the audio file
        // For demo purposes, we'll return a placeholder
        return 'https://example.com/audio/' . basename($session->audio_path);
    }

    public function getSignedImageUrl(CustomSession $session): ?string
    {
        if (!$session->image_path) {
            return null;
        }
        
        // In a real app, this would generate a signed URL for the image file
        return 'https://example.com/images/' . basename($session->image_path);
    }
}
