<?php

namespace App\Services;

use App\Models\CustomRequest;
use App\Models\CustomSession;
use App\Jobs\GenerateCustomSession;
use Illuminate\Support\Str;

class CustomSessionService
{
    public function queueSessionGeneration(CustomRequest $customRequest): void
    {
        // Create a placeholder session record
        $session = CustomSession::create([
            'user_id' => $customRequest->user_id,
            'custom_request_id' => $customRequest->id,
            'title' => $customRequest->topic,
            'description' => $customRequest->description,
            'voice_type' => $customRequest->voice_preference,
            'status' => 'queued',
        ]);

        // Update the custom request status
        $customRequest->update(['status' => 'approved']);

        // Queue the generation job
        GenerateCustomSession::dispatch($session);
    }

    public function generateSession(CustomSession $session): void
    {
        try {
            // Update status to generating
            $session->update(['status' => 'generating']);

            // Simulate AI/TTS generation (in real app, this would call external services)
            $this->simulateGeneration($session);

            // Update status to ready
            $session->update([
                'status' => 'ready',
                'generated_at' => now(),
                'generation_metadata' => [
                    'generation_time' => now()->toISOString(),
                    'model_used' => 'demo-ai-model',
                    'voice_engine' => 'demo-tts-engine',
                ],
            ]);

            // Update the custom request status
            $session->customRequest->update(['status' => 'completed']);

        } catch (\Exception $e) {
            // Handle generation failure
            $session->update([
                'status' => 'failed',
                'failed_at' => now(),
                'failure_reason' => $e->getMessage(),
            ]);

            // Log the error
            \Log::error('Custom session generation failed', [
                'session_id' => $session->id,
                'error' => $e->getMessage(),
            ]);
        }
    }

    private function simulateGeneration(CustomSession $session): void
    {
        // Simulate processing time
        sleep(2);

        // Generate fake transcript based on the request
        $transcript = $this->generateFakeTranscript($session->customRequest);

        // Generate fake audio file path (in real app, this would be the actual generated file)
        $filePath = 'custom-sessions/' . $session->id . '/session-' . Str::random(8) . '.mp3';

        // Update session with generated content
        $session->update([
            'transcript' => $transcript,
            'duration' => $session->customRequest->session_length,
            'file_path' => $filePath,
            'file_size' => rand(1000000, 5000000), // 1-5 MB
        ]);
    }

    private function generateFakeTranscript(CustomRequest $request): string
    {
        $topic = $request->topic;
        $length = $request->session_length;
        $voice = $request->voice_preference;

        $transcripts = [
            'stress-relief' => "Welcome to your personalized stress relief session. Take a deep breath in through your nose, and exhale slowly through your mouth. Feel the tension melting away as we focus on {$topic}. Let your mind drift to a place of peace and tranquility...",
            'sleep' => "As you prepare for sleep, let your body sink into the mattress. Feel the weight of the day lifting from your shoulders. We'll explore {$topic} in a gentle, soothing way that will help you drift into peaceful slumber...",
            'confidence' => "Today we're going to build your confidence and inner strength. Close your eyes and imagine yourself feeling powerful and capable. As we work with {$topic}, you'll discover reserves of confidence you never knew you had...",
            'default' => "Welcome to your personalized wellness session. Today we'll be exploring {$topic} in a way that's tailored just for you. Take a moment to settle into a comfortable position and let your mind become still...",
        ];

        $key = 'default';
        foreach ($transcripts as $pattern => $text) {
            if (stripos($topic, $pattern) !== false) {
                $key = $pattern;
                break;
            }
        }

        $baseTranscript = $transcripts[$key];
        
        // Add more content based on session length
        $additionalContent = "Continue breathing deeply and naturally. Feel the rhythm of your breath as it guides you deeper into this state of relaxation. {$topic} is becoming clearer to you now, and you're finding new insights and understanding...";
        
        // Repeat content to fill the session length
        $repetitions = max(1, floor($length / 60)); // Repeat every minute
        $fullTranscript = $baseTranscript;
        
        for ($i = 1; $i < $repetitions; $i++) {
            $fullTranscript .= " " . $additionalContent;
        }
        
        $fullTranscript .= " Now, as we bring this session to a close, take a moment to acknowledge the work we've done together. You're taking positive steps toward your wellness goals. When you're ready, gently bring your awareness back to the room, feeling refreshed and renewed.";

        return $fullTranscript;
    }

    public function getSignedUrl(CustomSession $session): string
    {
        // In a real app, this would generate a signed URL for the media file
        // For demo purposes, return a placeholder
        return route('custom.sessions.show', $session) . '?audio=true';
    }
}
