<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
		// =========================
		// Consents (legal)
		// =========================
		Schema::create('consents', function (Blueprint $table) {
			$table->uuid('id')->primary();
			$table->unsignedBigInteger('user_id');
			$table->enum('type', ['terms', 'privacy', 'wellness_disclaimer', 'marketing']);
			$table->string('version');
			$table->timestampTz('accepted_at');
			$table->unique(['user_id', 'type', 'version']);
			$table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
		});

		// =========================
		// Profile & Memory (Summaries)
		// =========================
		Schema::create('profiles', function (Blueprint $table) {
			$table->unsignedBigInteger('user_id')->primary();
			$table->string('display_name')->nullable();
			$table->json('preferences');
			$table->json('sensitivity');
			$table->timestampTz('created_at')->useCurrent();
			$table->timestampTz('updated_at')->useCurrent()->useCurrentOnUpdate();
			$table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
		});

		Schema::create('user_checkins', function (Blueprint $table) {
			$table->uuid('id')->primary();
			$table->unsignedBigInteger('user_id');
			$table->text('note');
			$table->timestampTz('created_at')->useCurrent();
			$table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
			$table->index(['user_id', 'created_at'], 'user_checkins_user_created_idx');
		});

		Schema::create('ai_context_summaries', function (Blueprint $table) {
			$table->uuid('id')->primary();
			$table->unsignedBigInteger('user_id');
			$table->text('summary');
			$table->json('source_window')->nullable();
			$table->timestampTz('created_at')->useCurrent();
			$table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
			$table->index(['user_id', 'created_at'], 'ai_context_summaries_user_created_idx');
		});

		// =========================
		// Library (Pre-Recorded Only; tags-only taxonomy)
		// =========================
		Schema::create('voice_actors', function (Blueprint $table) {
			$table->bigIncrements('id');
			$table->string('name');
			$table->enum('gender', ['female', 'male', 'nonbinary', 'prefer_not_to_say', 'other'])->nullable();
			// Storing tags array as JSON for cross-DB compatibility
			$table->json('tonality_tags');
		});

		Schema::create('tracks', function (Blueprint $table) {
			$table->uuid('id')->primary();
			$table->string('title');
			$table->text('description');
			$table->text('image_url')->nullable();
			$table->integer('duration_sec');
			$table->enum('level', ['beginner', 'intermediate', 'advanced']);
			$table->unsignedBigInteger('voice_actor_id')->nullable();
			$table->text('audio_url');
			$table->text('transcript_url')->nullable();
			$table->timestampTz('published_at')->nullable();
			$table->timestampTz('created_at')->useCurrent();
			$table->timestampTz('updated_at')->useCurrent()->useCurrentOnUpdate();
			$table->foreign('voice_actor_id')->references('id')->on('voice_actors');
			$table->index('published_at', 'tracks_published_idx');
		});

		Schema::create('tags', function (Blueprint $table) {
			$table->bigIncrements('id');
			$table->string('name')->unique();
			$table->string('slug')->unique();
		});

		Schema::create('track_tags', function (Blueprint $table) {
			$table->uuid('track_id');
			$table->unsignedBigInteger('tag_id');
			$table->primary(['track_id', 'tag_id']);
			$table->foreign('track_id')->references('id')->on('tracks')->onDelete('cascade');
			$table->foreign('tag_id')->references('id')->on('tags')->onDelete('cascade');
		});

		// =========================
		// Subscriptions / Entitlements
		// =========================
		Schema::create('subscriptions', function (Blueprint $table) {
			$table->uuid('id')->primary();
			$table->unsignedBigInteger('user_id');
			$table->enum('store', ['ios']);
			$table->string('product_id');
			$table->enum('status', ['active', 'cancelled', 'in_grace']);
			$table->timestampTz('period_end_at');
			$table->text('latest_receipt')->nullable();
			$table->timestampTz('created_at')->useCurrent();
			$table->timestampTz('updated_at')->useCurrent()->useCurrentOnUpdate();
			$table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
			$table->index('user_id', 'subscriptions_user_idx');
			$table->index(['status', 'period_end_at'], 'subscriptions_status_period_idx');
		});

		Schema::create('entitlements', function (Blueprint $table) {
			$table->unsignedBigInteger('user_id')->primary();
			$table->enum('library_access', ['free', 'premium']);
			$table->timestampTz('entitlement_expires_at')->nullable();
			$table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
		});

		Schema::create('webhooks', function (Blueprint $table) {
			$table->uuid('id')->primary();
			$table->string('provider');
			$table->string('event');
			$table->json('payload');
			$table->timestampTz('processed_at')->nullable();
			$table->string('status')->default('pending');
			$table->timestampTz('created_at')->useCurrent();
		});

		// =========================
		// Safety / Moderation (Compliance)
		// =========================
		Schema::create('safety_events', function (Blueprint $table) {
			$table->uuid('id')->primary();
			$table->unsignedBigInteger('user_id')->nullable();
			$table->enum('context', ['onboarding', 'checkin', 'ai_input', 'ai_output', 'script']);
			$table->string('rule_code');
			$table->enum('action', ['allow', 'redact', 'block', 'route_resources']);
			$table->text('excerpt')->nullable();
			$table->timestampTz('created_at')->useCurrent();
			$table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
			$table->index(['user_id', 'created_at'], 'safety_events_user_created_idx');
		});

		Schema::create('content_flags', function (Blueprint $table) {
			$table->uuid('id')->primary();
			$table->unsignedBigInteger('user_id')->nullable();
			$table->string('item_type');
			$table->uuid('item_id');
			$table->text('reason');
			$table->timestampTz('created_at')->useCurrent();
			$table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
		});

		// =========================
		// Engagement
		// =========================
		Schema::create('favorites', function (Blueprint $table) {
			$table->unsignedBigInteger('user_id');
			$table->uuid('track_id');
			$table->timestampTz('created_at')->useCurrent();
			$table->primary(['user_id', 'track_id']);
			$table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
			$table->foreign('track_id')->references('id')->on('tracks')->onDelete('cascade');
		});

		Schema::create('playback_events', function (Blueprint $table) {
			$table->bigIncrements('id');
			$table->unsignedBigInteger('user_id');
			$table->uuid('track_id');
			$table->enum('event', ['play', 'complete']);
			$table->timestampTz('at')->useCurrent();
			$table->foreign('user_id')->references('id')->on('users')->onDelete('cascade');
			$table->foreign('track_id')->references('id')->on('tracks')->onDelete('cascade');
			$table->index(['user_id', 'at'], 'playback_user_at_idx');
			$table->index(['track_id', 'at'], 'playback_track_at_idx');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
		Schema::dropIfExists('playback_events');
		Schema::dropIfExists('favorites');
		Schema::dropIfExists('content_flags');
		Schema::dropIfExists('safety_events');
		Schema::dropIfExists('webhooks');
		Schema::dropIfExists('entitlements');
		Schema::dropIfExists('subscriptions');
		Schema::dropIfExists('track_tags');
		Schema::dropIfExists('tags');
		Schema::dropIfExists('tracks');
		Schema::dropIfExists('voice_actors');
		Schema::dropIfExists('ai_context_summaries');
		Schema::dropIfExists('user_checkins');
		Schema::dropIfExists('profiles');
		Schema::dropIfExists('consents');
    }
};
