<?php

namespace App\Livewire\User;

use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Validation\Rules\File;
use App\Models\Department;
use App\Models\Account;
use App\Models\Receipt;
use App\Models\ReceiptLine;
use Illuminate\Support\Facades\Storage;
use App\Services\ReceiptProcessingService;
use App\Services\ImageResizeService;
use App\Services\FileProcessingService;
use App\Services\ParallelProcessingService;

class ReceiptUpload extends Component
{
    use WithFileUploads;

    public $receipt_date;
    public $merchant_name;
    public $total_amount;
    public $currency = 'GBP';
    public $department_id;
    public $account_id;
    public $is_personal = false;
    public $image;
    public $file;
    public $lines = [
        ['description' => '', 'quantity' => 1, 'unit_price' => null, 'line_total' => null],
    ];

    public function addLine()
    {
        $this->lines[] = ['description' => '', 'quantity' => 1, 'unit_price' => null, 'line_total' => null];
    }

    public function removeLine($index)
    {
        unset($this->lines[$index]);
        $this->lines = array_values($this->lines);
    }

    protected function rules(): array
    {
        return [
            'receipt_date' => ['required', 'date'],
            'merchant_name' => ['nullable', 'string', 'max:255'],
            'total_amount' => ['required', 'numeric'],
            'currency' => ['required', 'string', 'size:3'],
            'department_id' => ['nullable', 'exists:departments,id'],
            'account_id' => ['nullable', 'exists:accounts,id'],
            'is_personal' => ['boolean'],
            'image' => ['nullable', File::image()->max('10mb')],
            'file' => ['nullable', File::types(['eml', 'pdf'])->max('10mb')],
            'lines' => ['nullable', 'array'],
            'lines.*.description' => ['required_with:lines', 'string'],
            'lines.*.quantity' => ['nullable', 'integer', 'min:1'],
            'lines.*.unit_price' => ['nullable', 'numeric'],
            'lines.*.line_total' => ['required_with:lines', 'numeric'],
        ];
    }

    public function submit(ReceiptProcessingService $processingService, ImageResizeService $imageResize, FileProcessingService $fileProcessor, ParallelProcessingService $parallelProcessor)
    {
        $this->validate();

        $disk = config('filesystems.default');
        $imagePath = null;
        $imageHash = null;
        $fileData = null;
        
        if ($this->image) {
            // Generate a unique filename
            $filename = uniqid() . '.' . $this->image->getClientOriginalExtension();
            $imagePath = 'receipts/' . $filename;
            
            // Resize and store the image
            $imageResize->resizeAndStore($this->image, $imagePath, $disk);
            $imageHash = hash_file('sha256', Storage::disk($disk)->path($imagePath));
        } elseif ($this->file) {
            // Process EML or PDF file
            $fileData = $fileProcessor->processFile($this->file, $disk);
            
            // For EML files, use the generated PDF as the image
            if ($fileData['type'] === 'eml') {
                $imagePath = $fileData['pdf_path'];
                $imageHash = hash_file('sha256', Storage::disk($disk)->path($imagePath));
            } elseif ($fileData['type'] === 'pdf') {
                $imagePath = $fileData['pdf_path'];
                $imageHash = hash_file('sha256', Storage::disk($disk)->path($imagePath));
            }
        }

        $receipt = Receipt::create([
            'user_id' => auth()->id(),
            'receipt_date' => $this->receipt_date,
            'merchant_name' => $this->merchant_name,
            'total_amount' => $this->total_amount,
            'currency' => $this->currency,
            'image_disk' => $disk,
            'image_path' => $imagePath,
            'image_hash' => $imageHash,
            'department_id' => $this->department_id,
            'account_id' => $this->account_id,
            'is_personal' => (bool)$this->is_personal,
        ]);

        foreach ($this->lines ?? [] as $line) {
            if (!trim((string)($line['description'] ?? ''))) continue;
            ReceiptLine::create([
                'receipt_id' => $receipt->id,
                'description' => $line['description'],
                'quantity' => $line['quantity'] ?? 1,
                'unit_price' => $line['unit_price'] ?? 0,
                'line_total' => $line['line_total'] ?? 0,
            ]);
        }

        $receipt->load('lines');
        $parallelProcessor->processReceiptsOptimized([$receipt->id]);

        session()->flash('status', 'Receipt uploaded successfully.');
        return redirect()->route('user.receipts.upload');
    }

    public function render()
    {
        return view('livewire.user.receipt-upload', [
            'departments' => Department::orderBy('name')->get(),
            'accounts' => Account::orderBy('name')->get(),
        ]);
    }
}
