from flask import Flask, render_template, request, send_file, session
import pandas as pd
import os
from io import BytesIO
import secrets

app = Flask(__name__)
app.config['UPLOAD_FOLDER'] = 'uploads'
app.config['MAX_CONTENT_LENGTH'] = 16 * 1024 * 1024  # 16MB max upload size
app.config['SECRET_KEY'] = secrets.token_hex(16)  # Secret key for sessions

if not os.path.exists(app.config['UPLOAD_FOLDER']):
    os.makedirs(app.config['UPLOAD_FOLDER'])

@app.route('/')
def index():
    return render_template('index.html')

@app.route('/upload', methods=['POST'])
def upload_file():
    if 'file' not in request.files:
        return "No file part", 400
    file = request.files['file']
    if file.filename == '':
        return "No selected file", 400
    if file:
        try:
            df = pd.read_csv(file)
            # Identify duplicates across all columns
            duplicates = df[df.duplicated(keep=False)]
            
            # Store duplicates DataFrame in session as CSV string
            # This allows us to retrieve it later for download
            if not duplicates.empty:
                duplicates_csv = duplicates.to_csv(index=False)
                session['duplicates_csv'] = duplicates_csv
                session['has_duplicates'] = True
            else:
                session['has_duplicates'] = False
                session.pop('duplicates_csv', None)
            
            return render_template('results.html',
                                  original_data=df.to_html(classes='table table-striped', index=False),
                                  duplicate_data=duplicates.to_html(classes='table table-striped', index=False) if not duplicates.empty else None,
                                  has_duplicates=not duplicates.empty)
        except Exception as e:
            return f"Error processing file: {e}", 500
    return "Something went wrong", 500

@app.route('/download_duplicates', methods=['GET'])
def download_duplicates():
    # Check if duplicates exist in session
    if not session.get('has_duplicates', False) or 'duplicates_csv' not in session:
        return "No duplicates available for download. Please upload a CSV file first.", 404
    
    # Retrieve the duplicates CSV string from session
    duplicates_csv = session['duplicates_csv']
    
    # Create a BytesIO object from the CSV string
    output = BytesIO()
    output.write(duplicates_csv.encode('utf-8'))
    output.seek(0)
    
    # Send the file to the user
    return send_file(
        output,
        mimetype='text/csv',
        as_attachment=True,
        download_name='duplicates.csv'
    )

if __name__ == '__main__':
    app.run(debug=True)



