<?php

namespace App\Services;

use App\Models\Receipt;
use App\Services\OCR\ReceiptOCRService;
use App\Services\AI\ClassificationService;
use App\Services\AI\DiscountAnalysisService;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class ReceiptMergingService
{
    public function __construct(
        private ReceiptOCRService $ocr,
        private ClassificationService $classification,
        private DiscountAnalysisService $discountAnalysis
    ) {}

    /**
     * Merge multiple receipts into one
     */
    public function mergeReceipts(array $receiptIds): Receipt
    {
        if (count($receiptIds) < 2) {
            throw new \InvalidArgumentException('At least 2 receipts are required for merging');
        }

        $receipts = Receipt::whereIn('id', $receiptIds)->get();
        
        if ($receipts->count() !== count($receiptIds)) {
            throw new \InvalidArgumentException('Some receipts not found');
        }

        // Validate all receipts belong to the same user
        $userId = $receipts->first()->user_id;
        if ($receipts->pluck('user_id')->unique()->count() > 1) {
            throw new \InvalidArgumentException('All receipts must belong to the same user');
        }

        Log::info('Starting receipt merge', [
            'receipt_ids' => $receiptIds,
            'user_id' => $userId
        ]);

        // Create merged receipt
        $mergedReceipt = $this->createMergedReceipt($receipts);
        
        // Merge images
        $this->mergeImages($mergedReceipt, $receipts);
        
        // Re-run OCR on merged images
        $this->reprocessMergedReceipt($mergedReceipt);
        
        // Mark original receipts as merged
        $this->markReceiptsAsMerged($receipts, $mergedReceipt);
        
        Log::info('Receipt merge completed', [
            'merged_receipt_id' => $mergedReceipt->id,
            'original_receipt_ids' => $receiptIds
        ]);

        return $mergedReceipt;
    }

    /**
     * Create the merged receipt record
     */
    private function createMergedReceipt($receipts): Receipt
    {
        $firstReceipt = $receipts->first();
        
        // Determine the best date from all receipts
        $dates = $receipts->pluck('receipt_date')->filter()->sort();
        $bestDate = $dates->first() ?: null;
        
        // Determine the best merchant name (prefer non-null, then longest)
        $merchants = $receipts->pluck('merchant_name')->filter()->sortByDesc('length');
        $bestMerchant = $merchants->first() ?: 'Merged Receipt';
        
        // Sum all amounts
        $totalAmount = $receipts->sum('total_amount');
        
        // Use the most common currency
        $currencies = $receipts->pluck('currency')->filter()->countBy();
        $bestCurrency = $currencies->sortDesc()->keys()->first() ?: 'GBP';

        $mergedReceipt = Receipt::create([
            'user_id' => $firstReceipt->user_id,
            'receipt_date' => $bestDate,
            'merchant_name' => $bestMerchant,
            'total_amount' => $totalAmount,
            'currency' => $bestCurrency,
            'image_disk' => $firstReceipt->image_disk,
            'image_path' => null, // Will be set when we merge images
            'image_hash' => null, // Will be recalculated
            'department_id' => $firstReceipt->department_id,
            'account_id' => $firstReceipt->account_id,
            'is_personal' => $firstReceipt->is_personal,
            'meta' => [
                'is_merged' => true,
                'merged_from' => $receipts->pluck('id')->toArray(),
                'merge_date' => now()->toISOString(),
                'original_receipts' => $receipts->map(function($receipt) {
                    return [
                        'id' => $receipt->id,
                        'merchant_name' => $receipt->merchant_name,
                        'total_amount' => $receipt->total_amount,
                        'receipt_date' => $receipt->receipt_date?->toDateString(),
                    ];
                })->toArray()
            ]
        ]);

        return $mergedReceipt;
    }

    /**
     * Merge all receipt images into one
     */
    private function mergeImages(Receipt $mergedReceipt, $receipts): void
    {
        $imagePaths = $receipts->pluck('image_path')->filter()->toArray();
        
        if (empty($imagePaths)) {
            throw new \InvalidArgumentException('No images found in receipts to merge');
        }

        // Create a unique filename for the merged image
        $extension = pathinfo($imagePaths[0], PATHINFO_EXTENSION);
        $mergedFileName = 'merged_' . Str::uuid() . '.' . $extension;
        $mergedPath = 'receipts/' . $mergedFileName;

        // For now, we'll use the first image as the primary image
        // In a more sophisticated implementation, you could actually merge multiple images
        $firstImagePath = $imagePaths[0];
        $disk = $receipts->first()->image_disk;
        
        // Copy the first image as the merged image
        $imageContent = Storage::disk($disk)->get($firstImagePath);
        Storage::disk($disk)->put($mergedPath, $imageContent);
        
        // Calculate new hash
        $newHash = hash('sha256', $imageContent);
        
        // Update the merged receipt
        $mergedReceipt->update([
            'image_path' => $mergedPath,
            'image_hash' => $newHash
        ]);

        Log::info('Images merged', [
            'merged_receipt_id' => $mergedReceipt->id,
            'merged_path' => $mergedPath,
            'original_paths' => $imagePaths
        ]);
    }

    /**
     * Re-run OCR and classification on the merged receipt
     */
    private function reprocessMergedReceipt(Receipt $mergedReceipt): void
    {
        Log::info('Re-processing merged receipt with OCR', [
            'merged_receipt_id' => $mergedReceipt->id
        ]);

        // Re-run OCR to get the most complete information
        $this->ocr->extractReceiptData($mergedReceipt);
        $mergedReceipt->refresh();

        // Re-classify the receipt
        $this->classification->classifyReceipt($mergedReceipt);
        $this->classification->processLinesForClassification($mergedReceipt);
        
        // Re-analyze discounts
        $this->discountAnalysis->analyzeDiscounts($mergedReceipt);

        Log::info('Merged receipt re-processing completed', [
            'merged_receipt_id' => $mergedReceipt->id,
            'merchant_name' => $mergedReceipt->merchant_name,
            'total_amount' => $mergedReceipt->total_amount
        ]);
    }

    /**
     * Mark original receipts as merged
     */
    private function markReceiptsAsMerged($receipts, Receipt $mergedReceipt): void
    {
        foreach ($receipts as $receipt) {
            $meta = $receipt->meta ?? [];
            $meta['is_merged_into'] = $mergedReceipt->id;
            $meta['merge_date'] = now()->toISOString();
            
            $receipt->update([
                'meta' => $meta,
                'merchant_name' => $receipt->merchant_name . ' (Merged)',
                'total_amount' => 0, // Set to 0 to indicate it's been merged
            ]);
        }

        Log::info('Original receipts marked as merged', [
            'merged_receipt_id' => $mergedReceipt->id,
            'original_receipt_ids' => $receipts->pluck('id')->toArray()
        ]);
    }

    /**
     * Get merged receipt information
     */
    public function getMergedReceiptInfo(Receipt $receipt): ?array
    {
        if (!$receipt->meta || !($receipt->meta['is_merged'] ?? false)) {
            return null;
        }

        return [
            'is_merged' => true,
            'merged_from' => $receipt->meta['merged_from'] ?? [],
            'merge_date' => $receipt->meta['merge_date'] ?? null,
            'original_receipts' => $receipt->meta['original_receipts'] ?? []
        ];
    }

    /**
     * Check if a receipt is merged into another
     */
    public function isMergedInto(Receipt $receipt): ?int
    {
        return $receipt->meta['is_merged_into'] ?? null;
    }
}
