<?php

namespace App\Http\Livewire\Dashboard;

use App\Models\Departments;
use App\Models\Seasons;
use App\Models\ShipmentLine;
use Livewire\Component;
use App\Helper\Conversions;

class CustomerSalesByDepartment extends Component
{
    private const CACHE_VERSION = 3;
    public $selectedSeason = null;
    public $seasons = [];
    public $salesData = [];
    public $totalSales = 0;
    public $selectedOrderType = 'all';
    public $orderTypes = [
        'all' => 'All Orders',
        'wholesale' => 'Wholesale',
        'commission' => 'Commission'
    ];
    public $isLoading = true;

    public function mount()
    {
        $this->seasons = Seasons::orderBy('created_at', 'desc')->get();
        
        // Get saved season preference from user settings, or default to 4 seasons back
        $savedSeason = auth()->user()->setting('customer_sales_season_filter');
        if ($savedSeason && $this->seasons->contains('id', $savedSeason)) {
            $this->selectedSeason = $savedSeason;
        } else {
            // Default to 4 seasons back from the latest
            $this->selectedSeason = $this->seasons->skip(3)->first()?->id ?? $this->seasons->first()?->id;
        }
        
        // Don't load data immediately - let the page render first
    }

    public function updatedSelectedSeason()
    {
        // Save the season preference to user settings
        auth()->user()->settings(['customer_sales_season_filter' => $this->selectedSeason]);
        $this->isLoading = true;
        $this->loadSalesData();
        $this->isLoading = false;
    }

    public function updatedSelectedOrderType()
    {
        $this->isLoading = true;
        $this->loadSalesData();
        $this->isLoading = false;
    }

    public function refreshData()
    {
        $this->isLoading = true;
        $this->loadSalesData();
        $this->isLoading = false;
    }

    public function loadData()
    {
        $this->isLoading = true;
        $this->loadSalesData();
        $this->isLoading = false;
    }

    public function loadSalesData()
    {
        if (!$this->selectedSeason) {
            $this->salesData = [];
            $this->totalSales = 0;
            return;
        }

        // Resolve season model for currency conversion
        $seasonModel = $this->seasons->firstWhere('id', $this->selectedSeason) ?? Seasons::find($this->selectedSeason);

        // Create cache key (include version to invalidate old caches)
        $cacheKey = "customer_sales_dashboard_v" . self::CACHE_VERSION . "_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        
        // Try to get from cache first
        $cachedData = cache()->get($cacheKey);
        if ($cachedData) {
            $this->salesData = $cachedData['salesData'];
            $this->totalSales = $cachedData['totalSales'];
            return;
        }

        // Get shipment lines for the selected order type with pagination to prevent memory issues
        $shipmentLines = ShipmentLine::with([
            'customer_order_lines.customer_orders.departments',
            'customer_order_lines.customer_orders.customers',
            'customer_order_lines.colourways',
            'shipment_line_sizes.sizes',
            'customer_order_lines.customer_order_line_quantities'
        ])
        ->whereNull('shipment_lines.deleted_at')
        ->whereHas('customer_order_lines', function ($query) {
            $query->whereNull('customer_order_lines.deleted_at')
                  ->where('customer_order_lines.cancelled', false);
        })
        ->whereHas('customer_order_lines.customer_orders', function ($query) {
            $query->whereNull('deleted_at')
                  ->where('cancelled', false)
                  ->where('seasons_id', $this->selectedSeason);
            
            // Filter by order type if not 'all'
            if ($this->selectedOrderType !== 'all') {
                $query->where('order_type', $this->selectedOrderType);
            }
        })
        ->limit(1000) // Limit to prevent memory exhaustion
        ->get();

        // Debug: Log the total shipment lines found
        \Log::info("Dashboard: Found " . $shipmentLines->count() . " shipment lines for season {$this->selectedSeason}");

        $customerSales = [];

        foreach ($shipmentLines as $shipmentLine) {
            $customer = $shipmentLine->customer_order_lines->customer_orders->customers;
            $department = $shipmentLine->customer_order_lines->customer_orders->departments;
            $customerId = $customer->id;
            $customerName = $customer->name;
            $departmentName = $department->description;

            if (!isset($customerSales[$customerId])) {
                $customerSales[$customerId] = [
                    'name' => $customerName,
                    'department' => $departmentName,
                    'shipped_sales' => 0,
                    'remaining_sales' => 0,
                    'total_drops' => 0,
                    'shipped_drops' => 0
                ];
            }

            // Count this as a drop
            $customerSales[$customerId]['total_drops']++;
            
            // Count as shipped drop if complete
            if ($shipmentLine->complete) {
                $customerSales[$customerId]['shipped_drops']++;
            }

            // Calculate sales values based on complete vs incomplete drops
            foreach ($shipmentLine->shipment_line_sizes as $size) {
                // Skip deleted sizes
                if ($size->deleted_at) {
                    continue;
                }
                
                // Use shipped qty when the drop is complete, otherwise use ordered qty
                $orderedQty = $shipmentLine->complete && !empty($size->shipped_qty) ? $size->shipped_qty : $size->qty;
                
                if ($orderedQty > 0) {
                    $totalValue = 0;
                    $orderType = $shipmentLine->customer_order_lines->customer_orders->order_type;
                    
                    // Use the new price resolution system for both commission and wholesale orders
                    $colq = $shipmentLine->customer_order_lines->customer_order_line_quantities
                        ->firstWhere('sizes_id', $size->sizes_id);
                    
                    if ($colq) {
                        $priceModel = $colq->price_model;
                        
                        if ($priceModel) {
                            if ($orderType === 'commission') {
                                // For commission orders, use the price field and convert to base currency
                                $unitPrice = $colq->price ?? 0;
                                if ($unitPrice > 0) {
                                    $customerCurrency = $shipmentLine->customer_order_lines->customer_orders->customers->currency ?? '£';
                                    $totalValue = Conversions::convertCurrency($customerCurrency, '£', $unitPrice * $orderedQty, $seasonModel);
                                }
                            } else {
                                // For wholesale orders, use the cached quote_base from price resolution
                                $quoteBase = $priceModel['quote_base'] ?? 0;
                                $totalValue = $quoteBase * $orderedQty;
                            }
                        } else {
                            \Log::warning('Dashboard price model missing', [
                                'shipment_line_id' => $shipmentLine->id,
                                'size_id' => $size->sizes_id,
                                'customer' => $customerName,
                                'order_id' => $shipmentLine->customer_order_lines->customer_orders->id,
                                'order_type' => $orderType,
                            ]);
                        }
                    } else {
                        \Log::warning('Dashboard customer order line quantity missing', [
                            'shipment_line_id' => $shipmentLine->id,
                            'size_id' => $size->sizes_id,
                            'customer' => $customerName,
                            'order_id' => $shipmentLine->customer_order_lines->customer_orders->id,
                        ]);
                    }
                    
                    if ($totalValue > 0) {
                        // Debug: Log Mabli calculations
                        if (strtolower($customerName) === 'mabli') {
                            \Log::info("Mabli Drop {$shipmentLine->id}: Complete={$shipmentLine->complete}, Size={$size->sizes_id}, Qty={$orderedQty}, OrderType={$orderType}, Value={$totalValue}");
                        }
                        
                        // If drop is complete, add to shipped sales
                        if ($shipmentLine->complete) {
                            $customerSales[$customerId]['shipped_sales'] += $totalValue;
                        } else {
                            // If drop is not complete, add to remaining sales
                            $customerSales[$customerId]['remaining_sales'] += $totalValue;
                        }
                    }
                }
            }
        }

        // Calculate shipping percentage for each customer based on drops
        foreach ($customerSales as $customerId => &$data) {
            if ($data['total_drops'] > 0) {
                $data['shipping_percentage'] = ($data['shipped_drops'] / $data['total_drops']) * 100;
            } else {
                $data['shipping_percentage'] = 0;
            }
        }

        // Sort by shipping percentage descending (most complete at bottom)
        uasort($customerSales, function ($a, $b) {
            return $b['shipping_percentage'] <=> $a['shipping_percentage'];
        });

        $this->salesData = $customerSales;
        $this->totalSales = array_sum(array_column($customerSales, 'shipped_sales'));
        
        // Calculate additional stats for dashboard
        $pendingSales = array_sum(array_column($customerSales, 'remaining_sales'));
        $totalCustomers = count($customerSales);
        $avgProgress = $totalCustomers > 0 ? array_sum(array_column($customerSales, 'shipping_percentage')) / $totalCustomers : 0;
        
        // Emit stats to dashboard
        $this->emit('statsUpdated', [
            'totalSales' => $this->totalSales,
            'pendingSales' => $pendingSales,
            'totalCustomers' => $totalCustomers,
            'avgProgress' => round($avgProgress, 0)
        ]);
        
        // Debug: Log Mabli totals
        foreach ($customerSales as $customerId => $data) {
            if (strtolower($data['name']) === 'mabli') {
                \Log::info("Mabli Final: Shipped={$data['shipped_sales']}, Remaining={$data['remaining_sales']}, Drops={$data['shipped_drops']}/{$data['total_drops']}");
            }
        }
        
        // Cache the results for 24 hours
        cache()->put($cacheKey, [
            'salesData' => $this->salesData,
            'totalSales' => $this->totalSales
        ], now()->addHours(24));
    }

    public function render()
    {
        return view('livewire.dashboard.customer-sales-by-department');
    }
}
