<?php

namespace App\Http\Livewire\Dashboard;

use App\Models\Departments;
use App\Models\Seasons;
use App\Models\ShipmentLine;
use Livewire\Component;
use App\Helper\Conversions;
use Illuminate\Support\Facades\DB;

class CustomerSalesByDepartment extends Component
{
    private const CACHE_VERSION = 3;
    public $selectedSeason = null;
    public $seasons = [];
    public $salesData = [];
    public $totalSales = 0;
    public $selectedOrderType = 'all';
    public $orderTypes = [
        'all' => 'All Orders',
        'wholesale' => 'Wholesale',
        'commission' => 'Commission'
    ];
    public $isLoading = true;

    public function mount()
    {
        $this->seasons = Seasons::orderBy('created_at', 'desc')->get();
        
        // Get saved season preference from user settings, or default to 4 seasons back
        $savedSeason = auth()->user()->setting('customer_sales_season_filter');
        if ($savedSeason && $this->seasons->contains('id', $savedSeason)) {
            $this->selectedSeason = $savedSeason;
        } else {
            // Default to 4 seasons back from the latest
            $this->selectedSeason = $this->seasons->skip(3)->first()?->id ?? $this->seasons->first()?->id;
        }
        
        // Load data after mount
        $this->loadSalesData();
    }

    public function updatedSelectedSeason()
    {
        // Save the season preference to user settings
        auth()->user()->settings(['customer_sales_season_filter' => $this->selectedSeason]);
        $this->isLoading = true;
        $this->loadSalesData();
        $this->isLoading = false;
    }

    public function updatedSelectedOrderType()
    {
        $this->isLoading = true;
        $this->loadSalesData();
        $this->isLoading = false;
    }

    public function refreshData()
    {
        $this->isLoading = true;
        $this->loadSalesData();
        $this->isLoading = false;
    }

    public function loadData()
    {
        $this->isLoading = true;
        $this->loadSalesData();
        $this->isLoading = false;
    }

    public function loadSalesData()
    {
        if (!$this->selectedSeason) {
            $this->salesData = [];
            $this->totalSales = 0;
            return;
        }

        // Create cache key (include version to invalidate old caches)
        $cacheKey = "customer_sales_dashboard_v" . self::CACHE_VERSION . "_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        
        // Try to get from cache first
        $cachedData = cache()->get($cacheKey);
        if ($cachedData) {
            $this->salesData = $cachedData['salesData'];
            $this->totalSales = $cachedData['totalSales'];
            return;
        }

        // Use SQL to get aggregated sales data to prevent memory issues
        $query = DB::table('shipment_lines')
            ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
            ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
            ->join('customers', 'customer_orders.customers_id', '=', 'customers.id')
            ->join('departments', 'customer_orders.departments_id', '=', 'departments.id')
            ->join('shipment_line_sizes', 'shipment_lines.id', '=', 'shipment_line_sizes.shipment_line_id')
            ->whereNull('shipment_lines.deleted_at')
            ->whereNull('customer_order_lines.deleted_at')
            ->whereNull('customer_orders.deleted_at')
            ->where('customer_orders.cancelled', false)
            ->where('customer_order_lines.cancelled', false)
            ->where('customer_orders.seasons_id', $this->selectedSeason);

        // Filter by order type if not 'all'
        if ($this->selectedOrderType !== 'all') {
            $query->where('customer_orders.order_type', $this->selectedOrderType);
        }

        $salesData = $query->select([
                'customers.id as customer_id',
                'customers.name as customer_name',
                'departments.description as department_name',
                DB::raw('SUM(shipment_line_sizes.qty) as total_qty'),
                DB::raw('SUM(shipment_line_sizes.shipped_qty) as total_shipped_qty')
            ])
            ->groupBy('customers.id', 'customers.name', 'departments.id', 'departments.description')
            ->orderBy('total_qty', 'desc')
            ->limit(50) // Limit to top 50 customers
            ->get();

        // Debug: Log the total customers found
        \Log::info("Dashboard: Found " . $salesData->count() . " customers for season {$this->selectedSeason}");

        $customerSales = [];

        foreach ($salesData as $row) {
            $customerId = $row->customer_id;
            $customerName = $row->customer_name;
            $departmentName = $row->department_name;

            if (!isset($customerSales[$customerId])) {
                $customerSales[$customerId] = [
                    'name' => $customerName,
                    'department' => $departmentName,
                    'shipped_sales' => 0,
                    'remaining_sales' => 0,
                    'total_drops' => 0,
                    'shipped_drops' => 0
                ];
            }

            // Count this as a drop (simplified - we'll count unique customers)
            $customerSales[$customerId]['total_drops']++;
            
            // For shipped drops, we'll use a simple heuristic based on shipped quantity
            if ($row->total_shipped_qty > 0) {
                $customerSales[$customerId]['shipped_drops']++;
            }

            // Calculate sales values (simplified for now - we'll use basic quantity calculations)
            $totalQty = $row->total_qty ?? 0;
            $shippedQty = $row->total_shipped_qty ?? 0;
            $remainingQty = $totalQty - $shippedQty;
            
            // Use a simple average price estimate for now (this could be enhanced later)
            $estimatedPricePerUnit = 50; // This could be calculated from actual price data
            
            if ($shippedQty > 0) {
                $customerSales[$customerId]['shipped_sales'] += $shippedQty * $estimatedPricePerUnit;
            }
            
            if ($remainingQty > 0) {
                $customerSales[$customerId]['remaining_sales'] += $remainingQty * $estimatedPricePerUnit;
            }
        }

        // Calculate shipping percentage for each customer based on drops
        foreach ($customerSales as $customerId => &$data) {
            if ($data['total_drops'] > 0) {
                $data['shipping_percentage'] = ($data['shipped_drops'] / $data['total_drops']) * 100;
            } else {
                $data['shipping_percentage'] = 0;
            }
        }

        // Sort by shipping percentage descending (most complete at bottom)
        uasort($customerSales, function ($a, $b) {
            return $b['shipping_percentage'] <=> $a['shipping_percentage'];
        });

        $this->salesData = $customerSales;
        $this->totalSales = array_sum(array_column($customerSales, 'shipped_sales'));
        
        // Calculate additional stats for dashboard
        $pendingSales = array_sum(array_column($customerSales, 'remaining_sales'));
        $totalCustomers = count($customerSales);
        $avgProgress = $totalCustomers > 0 ? array_sum(array_column($customerSales, 'shipping_percentage')) / $totalCustomers : 0;
        
        // Emit stats to dashboard
        $this->emit('statsUpdated', [
            'totalSales' => $this->totalSales,
            'pendingSales' => $pendingSales,
            'totalCustomers' => $totalCustomers,
            'avgProgress' => round($avgProgress, 0)
        ]);
        
        // Cache the results for 24 hours
        cache()->put($cacheKey, [
            'salesData' => $this->salesData,
            'totalSales' => $this->totalSales
        ], now()->addHours(24));
    }

    public function render()
    {
        return view('livewire.dashboard.customer-sales-by-department');
    }
}
