<?php

namespace App\Console\Commands;

use App\Contracts\MemoryService;
use App\Contracts\SafetyService;
use App\Models\AiContextSummary;
use App\Models\User;
use App\Models\UserCheckin;
use Illuminate\Console\Command;
use Illuminate\Support\Str;

class MemorySummarize extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'memory:summarize {--user=* : Limit to specific user IDs}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create redacted AI context summaries from recent user check-ins and rotate history';

    public function handle(SafetyService $safety, MemoryService $memory): int
    {
        $userIds = $this->option('user');

        $usersQuery = User::query();
        if (!empty($userIds)) {
            $usersQuery->whereIn('id', $userIds);
        }

        $countSummarized = 0;

        $usersQuery->chunkById(200, function ($users) use ($safety, $memory, &$countSummarized) {
            foreach ($users as $user) {
                $lastAt = AiContextSummary::where('user_id', $user->id)->max('created_at');
                $since = $lastAt ? \Carbon\CarbonImmutable::parse($lastAt) : now()->subDays(30);
                $until = now();

                $notes = UserCheckin::where('user_id', $user->id)
                    ->whereBetween('created_at', [$since, $until])
                    ->orderBy('created_at')
                    ->pluck('note')
                    ->all();

                if (empty($notes)) {
                    continue;
                }

                // Simple summarization: concatenate bullet points and limit per note
                $joined = implode("\n- ", array_map(fn ($n) => Str::limit(trim((string) $n), 240), $notes));
                $summaryText = "Highlights:\n- " . $joined;

                // Redact sensitive output
                $summaryText = $safety->redact($summaryText);

                AiContextSummary::create([
                    'id' => (string) Str::uuid(),
                    'user_id' => $user->id,
                    'summary' => $summaryText,
                    'source_window' => [
                        'from' => $since->toDateString(),
                        'to' => $until->toDateString(),
                    ],
                    'created_at' => now(),
                ]);

                // Rotate to last 6 summaries
                $memory->rotateSummaries($user, 6);

                $countSummarized++;
            }
        });

        $this->info("Summarized {$countSummarized} users.");
        return self::SUCCESS;
    }
}


