<?php

namespace App\Jobs;

use App\Models\Receipt;
use App\Services\OCR\ReceiptOCRService;
use App\Services\Matching\ClassificationService;
use App\Services\Analysis\DiscountAnalysisService;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ProcessReceiptOCR implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $timeout = 60;
    public int $maxExceptions = 2;

    public function __construct(public int $receiptId) {}

    public function backoff(): array
    {
        return [15, 60, 300];
    }

    public function retryUntil(): \DateTimeInterface
    {
        return now()->addHours(6);
    }

    public function handle(
        ReceiptOCRService $ocr,
        ClassificationService $classification,
        DiscountAnalysisService $discountAnalysis
    ): void {
        try {
            $receipt = Receipt::with('lines')->find($this->receiptId);
            if (!$receipt) {
                return;
            }

            // Step 1: Extract data from the receipt image using OCR
            $ocr->extractReceiptData($receipt);
            
            // Refresh the receipt to get the updated data and lines
            $receipt->refresh();
            
            // Step 2: Classify the receipt and its lines (VAT analysis will be done later during finance submission)
            $classification->classifyReceipt($receipt);
            
            // Process receipt lines for classification only (no VAT analysis yet)
            $this->processLinesForClassification($receipt, $classification);
            
            // Analyze discounts and link them to line items
            $discountAnalysis->analyzeDiscounts($receipt);

        } catch (\Throwable $e) {
            \Log::warning('ProcessReceiptOCR: transient failure, will retry', [
                'receipt_id' => $this->receiptId,
                'error' => $e->getMessage(),
                'attempt' => method_exists($this, 'attempts') ? $this->attempts() : null,
            ]);
            $schedule = (array) $this->backoff();
            $index = max(0, min((int)($this->attempts() - 1), count($schedule) - 1));
            $this->release((int) $schedule[$index]);
        }
    }

    /**
     * Process receipt lines for classification only (no VAT analysis)
     * This is much faster as it batches all lines into a single API call
     */
    private function processLinesForClassification(Receipt $receipt, ClassificationService $classification): void
    {
        if ($receipt->lines->isEmpty()) {
            return;
        }

        // Batch classify all lines in a single API call for much better performance
        $classification->classifyReceiptLinesBatch($receipt->lines);
    }
}
