<?php

namespace App\Jobs;

use App\Models\Receipt;
use App\Services\OCR\ReceiptOCRService;
use App\Jobs\ProcessReceiptMatching;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ProcessReceiptOCR implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $timeout = 300; // 5 minutes, OCR can be slow
    public int $maxExceptions = 2;

    public function __construct(
        public int $receiptId,
        public ?int $userId = null,
        public bool $autoMatch = true,
        public int $logicalAttempt = 1
    ) {}

    public function backoff(): array
    {
        return [60, 300, 1800]; // 1 min, 5 mins, 30 mins
    }

    public function retryUntil(): \DateTimeInterface
    {
        return now()->addHours(12);
    }

    public function handle(ReceiptOCRService $ocrService): void
    {
        try {
            $receipt = Receipt::find($this->receiptId);
            if (!$receipt) {
                return;
            }

            $this->userId = $this->userId ?? $receipt->user_id;

            $ocrService->extractReceiptData($receipt);

            // Refresh to evaluate OCR quality
            $receipt->refresh();

            // If OCR output looks low-quality (unknown merchant and/or zero amount), retry up to 3 logical attempts
            if ($this->shouldRetryOcr($receipt)) {
                $attempt = max(1, (int) $this->logicalAttempt);
                if ($attempt < 3) {
                    \Log::info('ProcessReceiptOCR: low-quality result, scheduling retry', [
                        'receipt_id' => $this->receiptId,
                        'attempt' => $attempt,
                        'queue_attempts' => method_exists($this, 'attempts') ? $this->attempts() : null,
                        'merchant' => $receipt->merchant_name,
                        'total' => (float) $receipt->total_amount,
                        'retry_strategy' => 'enqueue_new_job',
                    ]);
                    $delay = $this->getBackoffDelayForAttempt($attempt);
                    self::dispatch($this->receiptId, $this->userId, $this->autoMatch, $attempt + 1)
                        ->delay(now()->addSeconds($delay))
                        ->onQueue('ocr');
                    return;
                }
            }

            // Optionally trigger matching immediately after successful OCR
            if ($this->autoMatch === true) {
                ProcessReceiptMatching::dispatch($this->receiptId, $this->userId)
                    ->onQueue('matching');
            }

        } catch (\Throwable $e) {
            \Log::warning('ProcessReceiptOCR: transient failure, will retry', [
                'receipt_id' => $this->receiptId,
                'error' => $e->getMessage(),
                'attempt' => $this->attempts(),
            ]);

            $backoff = $this->backoff();
            $attempt = $this->attempts() - 1;
            $delay = $backoff[$attempt] ?? end($backoff);

            $this->release($delay);
        }
    }

    private function shouldRetryOcr(\App\Models\Receipt $receipt): bool
    {
        $merchant = (string) ($receipt->merchant_name ?? '');
        $merchantTrimmed = trim($merchant);
        $unknown = ($merchantTrimmed === '') || (preg_match('/^\s*unknown(\s+merchant)?/i', $merchantTrimmed) === 1);
        $zeroAmount = ((float) $receipt->total_amount) <= 0.0;
        return $unknown || $zeroAmount;
    }

    private function getBackoffDelayForAttempt(int $attempt): int
    {
        $schedule = (array) $this->backoff();
        return (int) ($schedule[$attempt - 1] ?? end($schedule));
    }
}
