<?php

namespace App\Jobs;

use App\Models\Receipt;
use App\Services\OCR\ReceiptOCRService;
use App\Jobs\ProcessReceiptMatching;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ProcessReceiptOCR implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $timeout = 300; // 5 minutes, OCR can be slow
    public int $maxExceptions = 2;

    public function __construct(
        public int $receiptId,
        public ?int $userId = null,
        public bool $autoMatch = true
    ) {}

    public function backoff(): array
    {
        return [60, 300, 1800]; // 1 min, 5 mins, 30 mins
    }

    public function retryUntil(): \DateTimeInterface
    {
        return now()->addHours(12);
    }

    public function handle(ReceiptOCRService $ocrService): void
    {
        try {
            $receipt = Receipt::find($this->receiptId);
            if (!$receipt) {
                return;
            }

            $this->userId = $this->userId ?? $receipt->user_id;

            $ocrService->extractReceiptData($receipt);

            // Refresh to evaluate OCR quality
            $receipt->refresh();

            // If OCR output looks low-quality (unknown merchant and zero amount), retry up to 3 attempts
            if ($this->shouldRetryOcr($receipt)) {
                $attempt = max(1, (int) $this->attempts());
                if ($attempt < 3) {
                    \Log::info('ProcessReceiptOCR: low-quality result, retrying', [
                        'receipt_id' => $this->receiptId,
                        'attempt' => $attempt,
                        'merchant' => $receipt->merchant_name,
                        'total' => (float) $receipt->total_amount,
                    ]);
                    $backoff = $this->backoff();
                    $delay = $backoff[$attempt - 1] ?? end($backoff);
                    $this->release($delay);
                    return;
                }
            }

            // Optionally trigger matching immediately after successful OCR
            if ($this->autoMatch === true) {
                ProcessReceiptMatching::dispatch($this->receiptId, $this->userId)
                    ->onQueue('matching');
            }

        } catch (\Throwable $e) {
            \Log::warning('ProcessReceiptOCR: transient failure, will retry', [
                'receipt_id' => $this->receiptId,
                'error' => $e->getMessage(),
                'attempt' => $this->attempts(),
            ]);

            $backoff = $this->backoff();
            $attempt = $this->attempts() - 1;
            $delay = $backoff[$attempt] ?? end($backoff);

            $this->release($delay);
        }
    }

    private function shouldRetryOcr(\App\Models\Receipt $receipt): bool
    {
        $unknown = empty($receipt->merchant_name) || str_starts_with((string) $receipt->merchant_name, 'Unknown Merchant');
        $zeroAmount = ((float) $receipt->total_amount) <= 0.0;
        return $unknown || $zeroAmount;
    }
}
