<?php

namespace App\Jobs;

use App\Models\Receipt;
use App\Services\OCR\ReceiptOCRService;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ProcessReceiptOCR implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $timeout = 300; // 5 minutes, OCR can be slow
    public int $maxExceptions = 2;

    public function __construct(
        public int $receiptId,
        public ?int $userId = null,
        public bool $autoMatch = true
    ) {}

    public function backoff(): array
    {
        return [60, 300, 1800]; // 1 min, 5 mins, 30 mins
    }

    public function retryUntil(): \DateTimeInterface
    {
        return now()->addHours(12);
    }

    public function handle(ReceiptOCRService $ocrService): void
    {
        try {
            $receipt = Receipt::find($this->receiptId);
            if (!$receipt) {
                return;
            }

            $this->userId = $this->userId ?? $receipt->user_id;

            $ocrService->extractReceiptData($receipt);

            // Optionally trigger matching immediately after successful OCR
            if ($this->autoMatch === true) {
                ProcessReceiptMatching::dispatch($this->receiptId, $this->userId)
                    ->onQueue('matching');
            }

        } catch (\Throwable $e) {
            \Log::warning('ProcessReceiptOCR: transient failure, will retry', [
                'receipt_id' => $this->receiptId,
                'error' => $e->getMessage(),
                'attempt' => $this->attempts(),
            ]);

            $backoff = $this->backoff();
            $attempt = $this->attempts() - 1;
            $delay = $backoff[$attempt] ?? end($backoff);

            $this->release($delay);
        }
    }
}
