<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use App\Models\Receipt;
use App\Models\ReceiptLine;
use Illuminate\Validation\Rules\File;
use App\Services\ReceiptProcessingService;
use App\Services\ImageResizeService;

class ReceiptController extends Controller
{
    public function store(Request $request, ReceiptProcessingService $processingService, ImageResizeService $imageResize)
    {
        $validated = $request->validate([
            'receipt_date' => ['required', 'date'],
            'merchant_name' => ['nullable', 'string', 'max:255'],
            'total_amount' => ['required', 'numeric'],
            'currency' => ['nullable', 'string', 'size:3'],
            'department_id' => ['nullable', 'exists:departments,id'],
            'account_id' => ['nullable', 'exists:accounts,id'],
            'is_personal' => ['nullable', 'boolean'],
            'lines' => ['nullable', 'array'],
            'lines.*.description' => ['required_with:lines', 'string'],
            'lines.*.quantity' => ['nullable', 'integer', 'min:1'],
            'lines.*.unit_price' => ['nullable', 'numeric'],
            'lines.*.line_total' => ['required_with:lines', 'numeric'],
            'image' => ['nullable', File::image()->max('10mb')],
        ]);

        $disk = config('filesystems.default');
        $imagePath = null;
        $imageHash = null;
        if ($request->hasFile('image')) {
            // Generate a unique filename
            $filename = uniqid() . '.' . $request->file('image')->getClientOriginalExtension();
            $imagePath = 'receipts/' . $filename;
            
            // Resize and store the image
            $imageResize->resizeAndStore($request->file('image'), $imagePath, $disk);
            $imageHash = hash_file('sha256', Storage::disk($disk)->path($imagePath));
        }

        $receipt = Receipt::create([
            'user_id' => $request->user()->id,
            'receipt_date' => $validated['receipt_date'],
            'merchant_name' => $validated['merchant_name'] ?? null,
            'total_amount' => $validated['total_amount'],
            'currency' => $validated['currency'] ?? 'GBP',
            'image_disk' => $disk,
            'image_path' => $imagePath,
            'image_hash' => $imageHash,
            'department_id' => $validated['department_id'] ?? null,
            'account_id' => $validated['account_id'] ?? null,
            'is_personal' => (bool)($validated['is_personal'] ?? false),
        ]);

        foreach ($validated['lines'] ?? [] as $line) {
            ReceiptLine::create([
                'receipt_id' => $receipt->id,
                'description' => $line['description'],
                'quantity' => $line['quantity'] ?? 1,
                'unit_price' => $line['unit_price'] ?? 0,
                'line_total' => $line['line_total'],
            ]);
        }

        $receipt->load('lines');
        $processingService->process($receipt);

        return response()->json([
            'id' => $receipt->id,
            'status' => 'stored',
        ], 201);
    }
}
