<?php

namespace App\Services\AI;

use App\Models\StatementTransaction;
use App\Models\Receipt;
use App\Models\Department;
use App\Models\Account;
use App\Services\AI\VertexClient;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class FinanceClassificationAgent
{
    public function __construct(private VertexClient $vertex) {}

    /**
     * Classify a statement transaction (department, account, and split suggestions)
     */
    public function classifyTransaction(StatementTransaction $transaction, bool $hasReceipt = false): array
    {
        $model = config('vertex.models.classification', 'gemini-2.0-flash-thinking-exp');
        
        // Build context
        $context = $this->buildContext($transaction, $hasReceipt);
        
        // Get user learning data
        $userLearning = $this->getUserLearningData($transaction->user_id);
        
        $system = $this->buildSystemPrompt($userLearning);
        $prompt = $this->buildClassificationPrompt($transaction, $context);
        
        try {
            $response = $this->vertex->generateContent($model, $prompt, $system, 0.3);
            $result = $this->parseClassificationResponse($response);
            
            // Store for learning if classification was made
            if (!empty($result['department_id']) || !empty($result['account_id'])) {
                $this->recordClassification($transaction, $result);
            }
            
            return $result;
        } catch (\Exception $e) {
            Log::error('Finance classification failed', [
                'transaction_id' => $transaction->id,
                'error' => $e->getMessage()
            ]);
            return [
                'department_id' => null,
                'account_id' => null,
                'confidence' => 0,
                'reasoning' => 'Classification failed: ' . $e->getMessage(),
                'should_split' => false,
                'suggested_splits' => []
            ];
        }
    }

    /**
     * Learn from user corrections
     */
    public function learnFromCorrection(StatementTransaction $transaction, array $correction): void
    {
        $key = "finance_learning:user:{$transaction->user_id}";
        $learning = Cache::get($key, ['corrections' => [], 'patterns' => []]);
        
        $learning['corrections'][] = [
            'timestamp' => now()->toIso8601String(),
            'merchant' => $transaction->merchant_name,
            'amount' => $transaction->amount,
            'original_department' => $correction['original_department_id'] ?? null,
            'corrected_department' => $correction['corrected_department_id'],
            'original_account' => $correction['original_account_id'] ?? null,
            'corrected_account' => $correction['corrected_account_id'],
            'location' => $transaction->merchant_city,
            'category' => $transaction->merchant_category,
        ];
        
        // Keep last 100 corrections
        if (count($learning['corrections']) > 100) {
            $learning['corrections'] = array_slice($learning['corrections'], -100);
        }
        
        // Update patterns
        $this->updateLearningPatterns($learning, $transaction, $correction);
        
        Cache::put($key, $learning, now()->addYear());
    }

    private function buildSystemPrompt(array $userLearning): string
    {
        $departments = Department::all();
        $accounts = Account::all();
        
        $deptList = $departments->map(fn($d) => "- {$d->code}: {$d->name}")->join("\n");
        $accountList = $accounts->map(fn($a) => "- {$a->code}: {$a->name}")->join("\n");
        
        $learningContext = '';
        if (!empty($userLearning['patterns'])) {
            $learningContext = "\n\nLEARNED USER PREFERENCES:\n";
            foreach ($userLearning['patterns'] as $pattern) {
                $learningContext .= "- {$pattern['description']}\n";
            }
        }
        
        return <<<PROMPT
You are an expert UK business expense classification AI agent. Your role is to intelligently categorize business expenses by assigning:
1. Department code (business unit responsible)
2. Account code (type of expense)
3. Whether the transaction should be split into multiple departments

AVAILABLE DEPARTMENTS:
{$deptList}

AVAILABLE ACCOUNTS:
{$accountList}

UK BUSINESS CONTEXT:
- Consider UK business practices and common expenses
- VAT implications are handled separately
- Foreign currency transactions indicate business travel
- Multiple line items on receipts may belong to different departments

SPLITTING LOGIC:
- Suggest splits when a receipt has items that clearly belong to different departments
- Example: Fuel (Operations) + Shop items (Staff welfare)
- Example: Conference registration (Training) + Hotel (Travel)
- Only suggest splits when there's clear evidence from receipt line items

CONFIDENCE LEVELS:
- HIGH (0.8-1.0): Clear merchant/category mapping, or strong user pattern match
- MEDIUM (0.5-0.79): Reasonable inference from context
- LOW (0.0-0.49): Uncertain, needs user review
{$learningContext}

OUTPUT FORMAT (JSON):
{
  "department_id": <number|null>,
  "account_id": <number|null>,
  "confidence": <0.0-1.0>,
  "reasoning": "<brief explanation>",
  "should_split": <boolean>,
  "suggested_splits": [
    {
      "department_id": <number>,
      "account_id": <number>,
      "amount": <number>,
      "description": "<item description>",
      "reasoning": "<why this split>"
    }
  ]
}
PROMPT;
    }

    private function buildClassificationPrompt(StatementTransaction $transaction, array $context): string
    {
        $hasReceipt = !empty($context['receipt']);
        $receiptInfo = '';
        
        if ($hasReceipt) {
            $receipt = $context['receipt'];
            $receiptInfo = "\n\nRECEIPT INFORMATION:";
            $receiptInfo .= "\n- Merchant: {$receipt['merchant']}";
            $receiptInfo .= "\n- Total: £{$receipt['total']}";
            $receiptInfo .= "\n- Date: {$receipt['date']}";
            
            if (!empty($receipt['lines'])) {
                $receiptInfo .= "\n- Line items:";
                foreach ($receipt['lines'] as $line) {
                    $receiptInfo .= "\n  * {$line['description']}: £{$line['amount']}";
                    if (!empty($line['category'])) {
                        $receiptInfo .= " ({$line['category']})";
                    }
                }
            }
            
            if (!empty($receipt['notes'])) {
                $receiptInfo .= "\n- Additional notes: {$receipt['notes']}";
            }
        }
        
        return <<<PROMPT
Classify this business expense:

TRANSACTION DETAILS:
- Merchant: {$transaction->merchant_name}
- Amount: £{$transaction->amount} ({$transaction->currency})
- Date: {$transaction->transaction_date->format('Y-m-d')}
- Location: {$transaction->merchant_city}, {$transaction->merchant_state}
- Category: {$transaction->merchant_category}
- MCC: {$transaction->mcc}
{$receiptInfo}

Please analyze this transaction and provide:
1. Best department assignment
2. Best account code assignment
3. Confidence in your classification
4. Whether this should be split (if receipt shows items for different departments)
5. Suggested splits with amounts if applicable

Consider the merchant type, location, amount, and receipt details (if available) to make an intelligent decision.
PROMPT;
    }

    private function buildContext(StatementTransaction $transaction, bool $hasReceipt): array
    {
        $context = [
            'merchant' => $transaction->merchant_name,
            'amount' => $transaction->amount,
            'location' => $transaction->merchant_city,
            'category' => $transaction->merchant_category,
        ];
        
        if ($hasReceipt) {
            $transaction->load('matches.receipt.lines');
            $receipt = $transaction->matches->first()?->receipt;
            
            if ($receipt) {
                $context['receipt'] = [
                    'merchant' => $receipt->merchant_name,
                    'total' => $receipt->total_amount,
                    'date' => $receipt->receipt_date?->format('Y-m-d'),
                    'lines' => $receipt->lines->map(function($line) {
                        $meta = is_array($line->meta) ? $line->meta : (is_string($line->meta) ? json_decode($line->meta, true) : []);
                        return [
                            'description' => $line->description,
                            'amount' => $line->line_total,
                            'category' => $meta['category'] ?? null,
                        ];
                    })->toArray(),
                    'notes' => $receipt->meta['additional_notes'] ?? null,
                ];
            }
        }
        
        return $context;
    }

    private function parseClassificationResponse(string $response): array
    {
        // Extract JSON from response
        if (preg_match('/\{[\s\S]*\}/', $response, $matches)) {
            $json = json_decode($matches[0], true);
            
            if ($json) {
                return [
                    'department_id' => $json['department_id'] ?? null,
                    'account_id' => $json['account_id'] ?? null,
                    'confidence' => $json['confidence'] ?? 0.5,
                    'reasoning' => $json['reasoning'] ?? '',
                    'should_split' => $json['should_split'] ?? false,
                    'suggested_splits' => $json['suggested_splits'] ?? [],
                ];
            }
        }
        
        return [
            'department_id' => null,
            'account_id' => null,
            'confidence' => 0,
            'reasoning' => 'Could not parse response',
            'should_split' => false,
            'suggested_splits' => []
        ];
    }

    private function getUserLearningData(?int $userId): array
    {
        if (!$userId) return ['corrections' => [], 'patterns' => []];
        
        $key = "finance_learning:user:{$userId}";
        return Cache::get($key, ['corrections' => [], 'patterns' => []]);
    }

    private function recordClassification(StatementTransaction $transaction, array $result): void
    {
        // Store classification for potential future learning
        $key = "finance_classification:{$transaction->id}";
        Cache::put($key, [
            'department_id' => $result['department_id'],
            'account_id' => $result['account_id'],
            'confidence' => $result['confidence'],
            'timestamp' => now()->toIso8601String(),
        ], now()->addMonths(6));
    }

    private function updateLearningPatterns(array &$learning, StatementTransaction $transaction, array $correction): void
    {
        // Extract patterns from corrections
        $merchant = strtolower($transaction->merchant_name ?? '');
        $category = strtolower($transaction->merchant_category ?? '');
        
        // Create or update merchant pattern
        $patterns = &$learning['patterns'];
        $merchantKey = "merchant:{$merchant}";
        
        if (!isset($patterns[$merchantKey])) {
            $patterns[$merchantKey] = [
                'type' => 'merchant',
                'value' => $merchant,
                'department_id' => $correction['corrected_department_id'],
                'account_id' => $correction['corrected_account_id'],
                'count' => 1,
                'description' => "Merchant '{$transaction->merchant_name}' → Dept {$correction['corrected_department_id']}, Account {$correction['corrected_account_id']}"
            ];
        } else {
            $patterns[$merchantKey]['count']++;
            // Update if consistent
            if ($patterns[$merchantKey]['department_id'] !== $correction['corrected_department_id']) {
                $patterns[$merchantKey]['department_id'] = $correction['corrected_department_id'];
            }
        }
        
        // Category pattern
        if ($category) {
            $categoryKey = "category:{$category}";
            if (!isset($patterns[$categoryKey])) {
                $patterns[$categoryKey] = [
                    'type' => 'category',
                    'value' => $category,
                    'department_id' => $correction['corrected_department_id'],
                    'account_id' => $correction['corrected_account_id'],
                    'count' => 1,
                    'description' => "Category '{$category}' → Dept {$correction['corrected_department_id']}, Account {$correction['corrected_account_id']}"
                ];
            } else {
                $patterns[$categoryKey]['count']++;
            }
        }
    }
}

