<?php

namespace App\Http\Controllers;

use App\Services\LibraryService;
use Illuminate\Http\Request;
use Illuminate\View\View;

class LibraryController extends Controller
{
    public function __construct(
        private LibraryService $libraryService
    ) {}

    public function index(Request $request): View
    {
        $filters = $request->only(['category_id', 'difficulty_level', 'voice_type', 'search', 'sort_by', 'sort_order']);
        $filters['per_page'] = 12;
        
        $categories = $this->libraryService->getCategories();
        $tracks = $this->libraryService->getTracks($filters);
        $popularTracks = $this->libraryService->getPopularTracks(6);
        $recentTracks = $this->libraryService->getRecentTracks(6);

        return view('library.index', compact('categories', 'tracks', 'popularTracks', 'recentTracks', 'filters'));
    }

    public function category(string $slug): View
    {
        $category = $this->libraryService->getCategoryBySlug($slug);
        
        if (!$category) {
            abort(404);
        }

        return view('library.category', compact('category'));
    }

    public function track(string $slug): View
    {
        $track = $this->libraryService->getTrackBySlug($slug);
        
        if (!$track) {
            abort(404);
        }

        // Increment play count
        $this->libraryService->incrementPlayCount($track);

        // Get related tracks
        $relatedTracks = $this->libraryService->searchTracks($track->title, 4);

        return view('library.track', compact('track', 'relatedTracks'));
    }

    public function search(Request $request): View
    {
        $query = $request->get('q', '');
        $results = collect();
        
        if ($query) {
            $results = $this->libraryService->searchTracks($query, 20);
        }

        return view('library.search', compact('results', 'query'));
    }
}
