<?php

namespace App\Http\Controllers;

use App\Http\Requests\AuthAppleRequest;
use App\Models\User;
use App\Services\Auth\AppleJwtVerifier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class AuthController extends Controller
{
    public function apple(AuthAppleRequest $request, AppleJwtVerifier $verifier)
    {
        $tokenString = (string) $request->string('identity_token');
        try {
            $claims = $verifier->verify($tokenString);
            $sub = $claims['sub'] ?? null;
            if (! $sub) {
                return response()->json(['message' => 'invalid_token'], 422);
            }
        } catch (\Throwable $e) {
            Log::warning('SIWA verify failed', ['error' => $e->getMessage()]);

            return response()->json(['message' => 'invalid_token'], 422);
        }

        $user = User::firstOrCreate(['apple_sub_id' => $sub], [
            'name' => $claims['name'] ?? 'User',
            'email' => $claims['email'] ?? null,
            'password_hash' => Hash::make(Str::random(32)),
        ]);

        $token = $user->createToken('mobile', expiresAt: now()->addMinutes((int) env('SANCTUM_EXPIRATION', 43200)));

        return response()->json([
            'token' => $token->plainTextToken,
            'token_type' => 'Bearer',
            'expires_at' => $token->accessToken->expires_at?->toISOString(),
        ], 201);
    }

    public function refresh(Request $request)
    {
        $user = $request->user();
        $request->user()->currentAccessToken()?->delete();
        $token = $user->createToken('mobile', expiresAt: now()->addMinutes((int) env('SANCTUM_EXPIRATION', 43200)));

        return response()->json([
            'token' => $token->plainTextToken,
            'token_type' => 'Bearer',
            'expires_at' => $token->accessToken->expires_at?->toISOString(),
        ]);
    }
}
